<?php
// FILE: public_html/news.php
// PURPOSE: Displays a full, paginated list of all published news articles.

require_once 'db_connect.php';
require_once 'admin/functions.php';

// --- Page Meta for SEO ---
$page_title = "News & Updates | Genowa Delivery Unit";
$page_description = "Latest updates, projects, and development news from Homa Bay County.";
$og_image_url = "https://www.genowa.co.ke/assets/news-default.jpg";

// --- Pagination Setup ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 9;
$offset = ($page - 1) * $limit;

$articles = [];
$total_articles = 0;

try {
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM news_articles WHERE status = 'Published'");
    $total_articles = $count_stmt->fetchColumn();
    $total_pages = ceil($total_articles / $limit);

    $sql = "SELECT a.id, a.title, a.slug, a.content, a.featured_image, a.created_at, u.name as author_name
            FROM news_articles a
            JOIN users u ON a.author_id = u.id
            WHERE a.status = 'Published'
            ORDER BY a.created_at DESC
            LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error_message = "Could not load news articles due to a technical issue.";
}

include 'includes/header.php';
?>

<div class="bg-brand-blue">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-16 text-center">
        <h1 class="text-4xl md:text-5xl font-extrabold text-white">News & Updates</h1>
        <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-300">
            Stay informed with the latest announcements, stories, and progress from the Genowa Delivery Unit.
        </p>
    </div>
</div>

<main class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
    <!-- Search bar -->
    <div class="mb-8">
        <form method="GET" action="news.php" class="flex items-center max-w-lg mx-auto">
            <input type="text" name="search" placeholder="Search news..." class="w-full p-3 rounded-l-md border border-gray-300 focus:outline-none focus:ring-2 focus:ring-brand-blue dark:bg-dark-card dark:text-white" value="<?php echo htmlspecialchars($_GET['search'] ?? ''); ?>">
            <button type="submit" class="px-4 py-3 bg-brand-blue text-white font-semibold rounded-r-md hover:bg-brand-purple transition">Search</button>
        </form>
    </div>

    <?php if (isset($db_error_message)): ?>
        <div class="text-center bg-red-100 p-4 rounded-lg">
            <p class="text-red-700"><?php echo e($db_error_message); ?></p>
        </div>
    <?php elseif (empty($articles)): ?>
        <div class="text-center bg-white dark:bg-dark-card p-12 rounded-lg shadow-md">
            <h2 class="text-xl font-bold text-gray-800 dark:text-white">No News Yet</h2>
            <p class="mt-2 text-gray-500 dark:text-gray-400">There are currently no articles to display. Please check back soon for updates.</p>
        </div>
    <?php else: ?>
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
            <?php foreach ($articles as $article): ?>
                <article class="fade-in-section group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                    <a href="article.php?slug=<?php echo e($article['slug']); ?>">
                        <div class="relative">
                            <img src="admin/uploads/<?php echo e($article['featured_image'] ?? 'placeholder.jpg'); ?>" alt="Featured image for <?php echo e($article['title']); ?>" class="h-56 w-full object-cover" onerror="this.onerror=null;this.src='https://placehold.co/600x400/1E3A8A/FFFFFF?text=Genowa+News';">
                        </div>
                        <div class="p-6 flex flex-col flex-grow">
                            <p class="text-sm text-gray-500 dark:text-gray-400"><?php echo date('F j, Y', strtotime($article['created_at'])); ?></p>
                            <h3 class="mt-2 text-xl font-bold text-brand-blue dark:text-white group-hover:text-brand-purple transition-colors duration-300">
                                <?php echo e($article['title']); ?>
                            </h3>
                            <div class="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-3 flex-grow">
                                <?php echo e(substr(strip_tags($article['content']), 0, 150)); ?>...
                            </div>
                            <div class="mt-4 text-sm font-semibold text-brand-orange group-hover:underline">
                                Read More &rarr;
                            </div>
                        </div>
                    </a>
                </article>
            <?php endforeach; ?>
        </div>

        <!-- Pagination -->
        <div class="mt-12 flex justify-between items-center">
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-dark-card hover:bg-gray-50 dark:hover:bg-gray-700">Previous</a>
            <?php else: ?>
                <span class="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-700 text-sm font-medium rounded-md text-gray-400 bg-gray-100 dark:bg-gray-800 cursor-not-allowed">Previous</span>
            <?php endif; ?>
            <div class="text-sm text-gray-700 dark:text-gray-400">
                Page <?php echo $page; ?> of <?php echo $total_pages; ?>
            </div>
            <?php if ($page < $total_pages): ?>
                <a href="?page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-300 bg-white dark:bg-dark-card hover:bg-gray-50 dark:hover:bg-gray-700">Next</a>
            <?php else: ?>
                <span class="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-700 text-sm font-medium rounded-md text-gray-400 bg-gray-100 dark:bg-gray-800 cursor-not-allowed">Next</span>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</main>

<!-- Back to Top Button -->
<button onclick="window.scrollTo({ top: 0, behavior: 'smooth' });"
        class="fixed bottom-6 right-6 p-3 rounded-full bg-brand-blue text-white shadow-lg hover:bg-brand-purple transition">
    ↑
</button>

<script>
// Fade-in animation
document.addEventListener('DOMContentLoaded', () => {
    const animatedElements = document.querySelectorAll('.fade-in-section');
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('is-visible');
            }
        });
    }, { threshold: 0.1 });
    animatedElements.forEach(el => observer.observe(el));
});
</script>

<?php include 'includes/footer.php'; ?>
