<?php
// FILE: public_html/map.php

require_once 'db_connect.php';
require_once 'admin/functions.php';

$projects_for_map = [];
$constituencies_for_filter = [];
$all_wards_for_filter = [];

try {
    // MODIFIED: Include ward_id for JS filtering
    $stmt = $pdo->query("
        SELECT 
            p.id, p.name, p.status, p.latitude, p.longitude, 
            w.id AS ward_id, w.name AS ward,
            c.id AS constituency_id 
        FROM 
            projects p
        LEFT JOIN 
            wards w ON p.ward_id = w.id
        LEFT JOIN
            constituencies c ON w.constituency_id = c.id
        WHERE 
            p.latitude IS NOT NULL AND p.longitude IS NOT NULL AND p.is_approved = 1
    ");
    $projects_for_map = $stmt->fetchAll();

    $constituency_stmt = $pdo->query("SELECT id, name FROM constituencies ORDER BY name ASC");
    $constituencies_for_filter = $constituency_stmt->fetchAll();

    $ward_stmt = $pdo->query("SELECT id, name, constituency_id FROM wards ORDER BY name ASC");
    $all_wards_for_filter = $ward_stmt->fetchAll();

} catch (PDOException $e) {
    $db_error_message = "Could not load project data for the map.";
}

include 'includes/header.php';
?>

<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css" />
<link rel="stylesheet" href="https://unpkg.com/leaflet.markercluster@1.5.3/dist/MarkerCluster.css" />
<link rel="stylesheet" href="https://unpkg.com/leaflet.markercluster@1.5.3/dist/MarkerCluster.Default.css" />
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
<script src="https://unpkg.com/leaflet.markercluster@1.5.3/dist/leaflet.markercluster.js"></script>

<style>
  #map { height: 75vh; border-radius: 0.5rem; box-shadow: 0 10px 15px -3px rgba(0,0,0,0.1); }
  .map-loader { 
    position: absolute; top: 40%; left: 50%; transform: translate(-50%, -50%);
    z-index: 1000; background: rgba(255,255,255,0.8); padding: 2em 3em; border-radius: 1em;
    display: none; font-weight: bold; font-size: 1.5em; color: #1e293b;
  }
</style>

<div class="bg-brand-blue">
  <div class="container mx-auto px-4 py-16 text-center">
    <h1 class="text-4xl font-extrabold text-white">Interactive Project Map</h1>
    <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-300">
        Filter projects by status, constituency, or ward to explore development in Homa Bay County.
    </p>
  </div>
</div>

<main class="container mx-auto px-4 py-12 relative">
  <div class="mb-4 grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 items-center bg-white dark:bg-dark-card p-4 rounded-lg shadow">
    <div>
      <label for="statusFilter" class="block text-xs mb-1 text-gray-500">Status</label>
      <select id="statusFilter" class="border rounded px-3 py-2 dark:bg-gray-700 dark:border-gray-600 w-full">
        <option value="">All Statuses</option>
        <option value="Ongoing">Ongoing</option>
        <option value="Completed">Completed</option>
        <option value="Planned">Planned</option>
        <option value="Stalled">Stalled</option>
      </select>
    </div>
    <div>
      <label for="constituencyFilter" class="block text-xs mb-1 text-gray-500">Constituency</label>
      <select id="constituencyFilter" class="border rounded px-3 py-2 dark:bg-gray-700 dark:border-gray-600 w-full">
          <option value="">All Constituencies</option>
          <?php foreach ($constituencies_for_filter as $constituency): ?>
              <option value="<?php echo e($constituency['id']); ?>"><?php echo e($constituency['name']); ?></option>
          <?php endforeach; ?>
      </select>
    </div>
    <div>
      <label for="wardFilter" class="block text-xs mb-1 text-gray-500">Ward</label>
      <select id="wardFilter" class="border rounded px-3 py-2 dark:bg-gray-700 dark:border-gray-600 w-full" disabled>
        <option value="">All Wards</option>
      </select>
    </div>
    <div>
      <label class="block text-xs mb-1 text-transparent">.</label>
      <button id="reset-filters" class="w-full bg-gray-200 dark:bg-gray-600 py-2 px-4 rounded text-sm font-medium hover:bg-gray-300">Reset Filters</button>
    </div>
  </div>

  <div id="filtered-count" class="text-right text-sm text-gray-600 mb-2"></div>
  <div class="map-loader" id="map-loader">Loading Map…</div>

  <?php if (isset($db_error_message)): ?>
    <div class="text-center bg-red-100 p-4 rounded-lg"><p class="text-red-700"><?php echo e($db_error_message); ?></p></div>
  <?php else: ?>
    <div id="map"></div>
  <?php endif; ?>
</main>

<script>
document.addEventListener('DOMContentLoaded', function () {
    // Loader
    const loader = document.getElementById('map-loader');
    loader.style.display = 'block';

    const map = L.map('map').setView([-0.523, 34.459], 9);
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', { maxZoom: 18, attribution: '© OpenStreetMap' }).addTo(map);

    const allProjects = <?php echo json_encode($projects_for_map); ?>;
    const allWards = <?php echo json_encode($all_wards_for_filter); ?>;
    const markers = L.markerClusterGroup();
    let allMarkers = [];
    let countyBoundary;

    // Create all markers ONCE on page load, store ward_id for each marker
    allProjects.forEach(project => {
        const statusColors = { 'Ongoing': '#3B82F6', 'Completed': '#16A34A', 'Planned': '#FBBF24', 'Stalled': '#DC2626' };
        const color = statusColors[project.status] || '#6B7280';
        const marker = L.circleMarker([project.latitude, project.longitude], {
            radius: 8, fillColor: color, color: '#fff', weight: 2, opacity: 1, fillOpacity: 0.9
        });
        const popupContent = `<div class='font-sans'><h3 class='font-bold mb-1'>${project.name}</h3><p class='text-sm'>Status: <strong>${project.status}</strong></p><p class='text-sm mb-1'>Ward: ${project.ward ?? 'Unknown'}</p><a href='project.php?id=${project.id}' class='text-brand-purple font-semibold hover:underline'>View Details →</a></div>`;
        marker.bindPopup(popupContent);
        marker.projectData = {
            ...project,
            // Make sure types are correct for comparison
            constituency_id: project.constituency_id ? parseInt(project.constituency_id) : null,
            ward_id: project.ward_id ? parseInt(project.ward_id) : null
        };
        allMarkers.push(marker);
    });

    // Get filter elements
    const statusFilter = document.getElementById('statusFilter');
    const constituencyFilter = document.getElementById('constituencyFilter');
    const wardFilter = document.getElementById('wardFilter');
    const resetBtn = document.getElementById('reset-filters');
    const countDisplay = document.getElementById('filtered-count');

    function populateWards(constituencyId) {
        wardFilter.innerHTML = '<option value="">All Wards</option>';
        if (!constituencyId) {
            wardFilter.disabled = true;
            return;
        }
        const relevantWards = allWards.filter(ward => ward.constituency_id == constituencyId);
        relevantWards.forEach(ward => {
            const option = document.createElement('option');
            option.value = ward.id; // Use ward.id not name
            option.textContent = ward.name;
            wardFilter.appendChild(option);
        });
        wardFilter.disabled = false;
    }

    // Filtering function
    function applyFilters() {
        const statusVal = statusFilter.value;
        const constituencyVal = constituencyFilter.value ? parseInt(constituencyFilter.value) : null;
        const wardVal = wardFilter.value ? parseInt(wardFilter.value) : null;

        markers.clearLayers();
        const filteredMarkers = allMarkers.filter(marker => {
            const data = marker.projectData;
            const statusMatch = (statusVal === '' || data.status === statusVal);
            const constituencyMatch = (constituencyVal === null || data.constituency_id === constituencyVal);
            const wardMatch = (wardVal === null || data.ward_id === wardVal);
            return statusMatch && constituencyMatch && wardMatch;
        });
        markers.addLayers(filteredMarkers);

        // Count feedback
        countDisplay.textContent = `Showing ${filteredMarkers.length} project${filteredMarkers.length === 1 ? '' : 's'}`;

        // Zoom logic
        if (filteredMarkers.length > 0) {
            const group = new L.featureGroup(filteredMarkers);
            map.fitBounds(group.getBounds().pad(0.2));
        } else if (countyBoundary) {
            map.fitBounds(countyBoundary.getBounds());
        }
    }
    
    constituencyFilter.addEventListener('change', () => {
        populateWards(constituencyFilter.value);
        applyFilters();
    });
    statusFilter.addEventListener('change', applyFilters);
    wardFilter.addEventListener('change', applyFilters);

    // --- BUG FIX: Reordered and simplified to prevent event cascade issues ---
    resetBtn.addEventListener('click', () => {
        statusFilter.value = '';
        wardFilter.value = ''; // Reset the dependent filter's value FIRST.
        constituencyFilter.value = ''; // NOW, changing this will trigger the 'change' event which handles the rest.
    });

    // Initial load
    markers.addLayers(allMarkers);
    map.addLayer(markers);
    countDisplay.textContent = `Showing ${allMarkers.length} projects`;

    // Load and display Homa Bay County boundary
    fetch('data/homa_bay_boundary.geojson')
      .then(res => res.json())
      .then(geojson => {
          countyBoundary = L.geoJSON(geojson, { style: { color: "#F97316", weight: 3, opacity: 0.8, fillOpacity: 0.05 } }).addTo(map);
          map.fitBounds(countyBoundary.getBounds());
          map.setMaxBounds(countyBoundary.getBounds().pad(0.5));
          map.setMinZoom(9);
          loader.style.display = 'none';
      }).catch(() => {
          loader.textContent = "Could not load county boundary.";
      });
});
</script>

<?php include 'includes/footer.php'; ?>