<?php
// FILE: public_html/article.php
// --- MODIFIED ---: Added the 'View Related Project' button code.

require_once 'db_connect.php';
require_once 'admin/functions.php';

// --- Get Article by ID or Slug ---
$article_id = $_GET['id'] ?? null;
$article_slug = $_GET['slug'] ?? null;

if (!$article_id && !$article_slug) {
    header("Location: news.php");
    exit;
}

$article = null;
$article_tags = [];
$more_news = [];
$read_time_minutes = 0;
$og_image_url = "https://www.genowa.co.ke/admin/assets/genowad.png";
$page_title = "Genowa Delivery Unit";
$page_description = "Tracking development projects and progress in Homa Bay County.";
$og_url = "";

// --- SQL condition for published/scheduled posts ---
$published_sql_condition = "(a.status = 'Published' OR (a.status = 'Scheduled' AND a.publish_at <= NOW()))";

// --- Fetch article data ---
try {
    $sql = "SELECT a.*, u.name as author_name, p.name as project_name, c.name as category_name, c.slug as category_slug
            FROM news_articles a
            JOIN users u ON a.author_id = u.id
            LEFT JOIN projects p ON a.project_id = p.id
            LEFT JOIN categories c ON a.category_id = c.id
            WHERE " . $published_sql_condition;

    if ($article_id) {
        $stmt = $pdo->prepare($sql . " AND a.id = :id");
        $stmt->execute(['id' => $article_id]);
    } else {
        $stmt = $pdo->prepare($sql . " AND a.slug = :slug");
        $stmt->execute(['slug' => $article_slug]);
    }

    $article = $stmt->fetch();

    if ($article) {
        // Fetch tags for the article
        $tags_stmt = $pdo->prepare("SELECT t.name, t.slug FROM tags t JOIN article_tags at ON t.id = at.tag_id WHERE at.article_id = :article_id");
        $tags_stmt->execute(['article_id' => $article['id']]);
        $article_tags = $tags_stmt->fetchAll();

        // Set meta tags and other page variables
        $page_title = e($article['title']) . " | Genowa Delivery Unit";
        $page_description = e(substr(strip_tags($article['content']), 0, 155));
        $og_url = "https://www.genowa.co.ke/article.php?slug=" . urlencode($article['slug']);

        if (!empty($article['featured_image'])) {
            $og_image_url = "https://www.genowa.co.ke/admin/uploads/" . e($article['featured_image']);
        }

        $word_count = str_word_count(strip_tags($article['content']));
        $read_time_minutes = ceil($word_count / 200);

        // Fetch "More News"
        $more_news_stmt = $pdo->prepare("SELECT a.id, a.title, a.slug, a.featured_image, a.created_at, u.name as author_name FROM news_articles a JOIN users u ON a.author_id = u.id WHERE " . $published_sql_condition . " AND a.id != :current_id ORDER BY a.created_at DESC LIMIT 3");
        $more_news_stmt->execute(['current_id' => $article['id']]);
        $more_news = $more_news_stmt->fetchAll();
    }

} catch (PDOException $e) {
    $db_error_message = "Could not load the article due to a technical issue.";
}

include 'includes/header.php';
?>

<?php if ($article): ?>
    <link rel="canonical" href="<?php echo $og_url; ?>">
    <meta property="og:title" content="<?php echo $page_title; ?>">
    <meta property="og:description" content="<?php echo $page_description; ?>">
    <meta property="og:image" content="<?php echo $og_image_url; ?>">
    <meta property="og:url" content="<?php echo $og_url; ?>">
    <meta name="twitter:card" content="summary_large_image">
    <script type="application/ld+json">
    {
        "@context": "https://schema.org",
        "@type": "NewsArticle",
        "headline": "<?php echo e($article['title']); ?>",
        "image": ["<?php echo $og_image_url; ?>"],
        "datePublished": "<?php echo date(DATE_ISO8601, strtotime($article['created_at'])); ?>",
        "dateModified": "<?php echo date(DATE_ISO8601, strtotime($article['updated_at'])); ?>",
        "articleSection": "<?php echo e($article['category_name'] ?? 'News'); ?>",
        "keywords": "<?php echo implode(', ', array_column($article_tags, 'name')); ?>",
        "author": {
            "@type": "Person",
            "name": "<?php echo e($article['author_name']); ?>"
        },
        "publisher": {
            "@type": "Organization",
            "name": "Genowa Delivery Unit",
            "logo": {
                "@type": "ImageObject",
                "url": "https://www.genowa.co.ke/admin/assets/genowad.png"
            }
        }
    }
    </script>
<?php endif; ?>

<script src="https://cdn.tailwindcss.com/3.4.1"></script>
<main class="bg-white dark:bg-dark-card">
<?php if (isset($db_error_message)): ?>
    <div class="text-center bg-red-100 p-4 my-8 container mx-auto rounded-lg">
        <p class="text-red-700"><?php echo e($db_error_message); ?></p>
    </div>
<?php elseif ($article): ?>
    <div class="relative">
        <div class="h-96 w-full bg-cover bg-center" style="background-image: url('admin/uploads/<?php echo e($article['featured_image'] ?? 'placeholder.jpg'); ?>');">
            <div class="absolute inset-0 bg-black/50"></div>
        </div>
        <div class="absolute inset-0 flex items-center justify-center">
            <div class="text-center text-white p-4 max-w-4xl mx-auto">
                <?php if (!empty($article['category_name'])): ?>
                <a href="category.php?slug=<?php echo e($article['category_slug']); ?>" class="inline-block bg-brand-orange text-white text-sm font-bold px-3 py-1 rounded-full uppercase mb-4 tracking-wider">
                    <?php echo e($article['category_name']); ?>
                </a>
                <?php endif; ?>
                <h1 class="text-4xl md:text-6xl font-extrabold text-shadow"><?php echo e($article['title']); ?></h1>
                <p class="mt-4 text-lg text-gray-200 text-shadow flex items-center justify-center space-x-2">
                    <span>Published on <?php echo date('F j, Y', strtotime($article['created_at'])); ?> by <?php echo e($article['author_name']); ?></span>
                    <span class="mx-2">&middot;</span>
                    <span>Approx. <?php echo $read_time_minutes; ?> min read</span>
                </p>
            </div>
        </div>
    </div>

    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-12">
        <div class="max-w-4xl mx-auto" id="print-section">
            <article class="prose prose-sm sm:prose lg:prose-lg xl:prose-xl dark:prose-invert max-w-none">
                <?php echo $article['content']; ?>
            </article>

            <?php if (!empty($article_tags)): ?>
            <div class="mt-12 pt-8 border-t dark:border-gray-700 no-print">
                <h3 class="text-lg font-bold text-gray-800 dark:text-white mb-4">Related Tags</h3>
                <div class="flex flex-wrap gap-2">
                    <?php foreach ($article_tags as $tag): ?>
                        <a href="tag.php?slug=<?php echo e($tag['slug']); ?>" class="bg-brand-light-gray dark:bg-dark-bg text-brand-blue dark:text-gray-300 text-sm font-medium px-3 py-1 rounded-full hover:bg-gray-300 dark:hover:bg-gray-700 transition-colors">
                            #<?php echo e($tag['name']); ?>
                        </a>
                    <?php endforeach; ?>
                </div>
            </div>
            <?php endif; ?>

            <?php if (!empty($article['project_id'])): ?>
            <div class="mt-12 pt-8 border-t dark:border-gray-700 text-center no-print">
                <a href="project.php?id=<?php echo e($article['project_id']); ?>" class="inline-flex items-center gap-x-2 rounded-md bg-brand-blue px-4 py-3 text-base font-semibold text-white shadow-sm hover:bg-brand-purple transition-transform duration-200 hover:scale-105">
                    <svg class="h-5 w-5" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M11.25 4.5l7.5 7.5-7.5 7.5m-6-15l7.5 7.5-7.5 7.5" /></svg>
                    View the Related Project: <?php echo e($article['project_name']); ?>
                </a>
            </div>
            <?php endif; ?>
            
        </div>
    </div>

    <?php if (!empty($more_news)): ?>
        <?php endif; ?>

<?php else: ?>
    <div class="text-center py-20 container mx-auto">
        <h1 class="text-3xl font-bold text-red-600">Article Not Found</h1>
        <p class="mt-4 text-gray-600">The article you are looking for may have been removed or is not yet published.</p>
        <a href="news.php" class="mt-6 inline-block bg-brand-orange text-white font-bold py-3 px-6 rounded-full hover:bg-opacity-90">Back to All News</a>
    </div>
<?php endif; ?>
</main>

<?php include 'includes/footer.php'; ?>