<?php
// FILE: admin/user_management.php
// PURPOSE: A page for SuperAdmins to manage users.

require_once 'functions.php';
require_once 'db.php';

// Security: Only Admins and SuperAdmins can access this page
require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}


$feedback_message = '';
$feedback_type = ''; // 'success' or 'error'

// --- Handle POST Actions (Approve, Delete, Update Role) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $user_id_to_action = $_POST['user_id'] ?? null;
    $current_user_id = $_SESSION['id'];

    if ($user_id_to_action) {
        // --- Action: Update Role ---
        if (isset($_POST['update_role'])) {
            $new_role = $_POST['role'] ?? '';
            // Prevent a user from changing their own role
            if ($user_id_to_action == $current_user_id) {
                $feedback_message = "For security, you cannot change your own role.";
                $feedback_type = 'error';
            } elseif (in_array($new_role, ['Inputer', 'Approver', 'Admin', 'SuperAdmin'])) {
                // Prevent Admin from creating another SuperAdmin
                if (!is_super_admin() && $new_role == 'SuperAdmin') {
                    $feedback_message = "You do not have permission to create SuperAdmins.";
                    $feedback_type = 'error';
                } else {
                    $stmt = $pdo->prepare("UPDATE users SET role = :role WHERE id = :id");
                    $stmt->execute(['role' => $new_role, 'id' => $user_id_to_action]);
                    $feedback_message = "User role updated successfully.";
                    $feedback_type = 'success';
                }
            } else {
                $feedback_message = "Invalid role selected.";
                $feedback_type = 'error';
            }
        }

        // --- Action: Approve User ---
        if (isset($_POST['approve_user'])) {
            // First, fetch user details to get their contact info
            $user_stmt = $pdo->prepare("SELECT name, email, phone_number FROM users WHERE id = :id");
            $user_stmt->execute(['id' => $user_id_to_action]);
            $user_to_notify = $user_stmt->fetch();

            if ($user_to_notify) {
                // Now, activate the user
                $stmt = $pdo->prepare("UPDATE users SET is_active = 1 WHERE id = :id");
                $stmt->execute(['id' => $user_id_to_action]);
                $feedback_message = 'User has been approved and is now active.';
                $feedback_type = 'success';

                // Send notifications
                $email_subject = "Your Genowa Account has been Approved!";
                $email_body = "<p>Dear " . e($user_to_notify['name']) . ",</p><p>Your account for the Genowa Delivery Unit platform has been approved. You can now log in using the credentials you provided during sign-up.</p><p>Thank you!</p>";
                send_email($user_to_notify['email'], $email_subject, $email_body);

                $sms_message = "Hello " . e($user_to_notify['name']) . ", your Genowa Delivery Unit account has been approved. You can now log in.";
                if (!empty($user_to_notify['phone_number'])) {
                    send_sms($user_to_notify['phone_number'], $sms_message);
                }
            } else {
                 $feedback_message = 'Could not find user to approve.';
                 $feedback_type = 'error';
            }
        }

        // --- Action: Delete User ---
        if (isset($_POST['delete_user'])) {
             if ($user_id_to_action == $current_user_id) {
                $feedback_message = "You cannot delete your own account.";
                $feedback_type = 'error';
            } else {
                 try {
                    $stmt = $pdo->prepare("DELETE FROM users WHERE id = :id");
                    $stmt->execute(['id' => $user_id_to_action]);
                    $feedback_message = 'User has been deleted successfully.';
                    $feedback_type = 'success';
                } catch (PDOException $e) {
                    $feedback_message = 'Could not delete user. They may have projects or other items associated with their account.';
                    $feedback_type = 'error';
                }
            }
        }
    }
}


// --- Fetch All Users for Display ---
try {
    // Join with wards table to get the ward name if applicable
    $stmt = $pdo->query("SELECT u.id, u.name, u.email, u.role, u.is_active, w.name as ward_name 
                         FROM users u 
                         LEFT JOIN wards w ON u.ward_id = w.id 
                         ORDER BY u.id ASC");
    $users = $stmt->fetchAll();
} catch (PDOException $e) {
    $db_error = "Error fetching users: " . $e->getMessage();
}

$roles = ['Inputer', 'Approver', 'Admin', 'SuperAdmin'];

include 'includes/header.php';
?>
<div class="min-h-full">
    <header class="bg-white shadow">
        <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8">
            <h1 class="text-3xl font-bold tracking-tight text-gray-900">User Management</h1>
        </div>
    </header>

    <main>
        <div class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
            <?php if ($feedback_message): ?>
                <div class="mb-4 p-4 rounded-md <?php echo $feedback_type === 'success' ? 'bg-green-100 border-l-4 border-green-500 text-green-700' : 'bg-red-100 border-l-4 border-red-500 text-red-700'; ?>">
                    <p><?php echo e($feedback_message); ?></p>
                </div>
            <?php endif; ?>

             <div class="overflow-x-auto bg-white rounded-lg shadow">
                <?php if (isset($db_error)): ?>
                    <div class="p-4" role="alert"><p><?php echo e($db_error); ?></p></div>
                <?php else: ?>
                    <table class="min-w-full divide-y divide-gray-200">
                        <thead class="bg-gray-50">
                            <tr>
                                <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">User</th>
                                <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Assigned Role</th>
                                <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                                <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Actions</th>
                            </tr>
                        </thead>
                        <tbody class="bg-white divide-y divide-gray-200">
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <div class="text-sm font-medium text-gray-900"><?php echo e($user['name']); ?></div>
                                        <div class="text-sm text-gray-500"><?php echo e($user['email']); ?></div>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <form method="POST" action="user_management.php" class="flex items-center space-x-2">
                                            <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                            <select name="role" class="block w-full rounded-md border-gray-300 shadow-sm focus:border-brand-purple focus:ring-brand-purple sm:text-sm" <?php if ($user['id'] == $_SESSION['id'] || (!is_super_admin() && $user['role'] == 'SuperAdmin')) echo 'disabled'; ?>>
                                                <?php foreach ($roles as $role): ?>
                                                    <?php if (!is_super_admin() && $role == 'SuperAdmin') continue; // Hide SuperAdmin option if not SuperAdmin ?>
                                                    <option value="<?php echo $role; ?>" <?php if ($user['role'] == $role) echo 'selected'; ?>><?php echo $role; ?></option>
                                                <?php endforeach; ?>
                                            </select>
                                            <button type="submit" name="update_role" class="text-sm text-brand-blue hover:underline" <?php if ($user['id'] == $_SESSION['id']) echo 'disabled'; ?>>Save</button>
                                        </form>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap">
                                        <?php if ($user['is_active']): ?>
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 text-green-800">Active</span>
                                        <?php else: ?>
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-yellow-100 text-yellow-800">Pending Approval</span>
                                        <?php endif; ?>
                                    </td>
                                    <td class="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                        <form method="POST" action="user_management.php" class="inline-flex space-x-3">
                                            <input type="hidden" name="user_id" value="<?php echo e($user['id']); ?>">
                                            <?php if (!$user['is_active']): ?>
                                                <button type="submit" name="approve_user" class="text-green-600 hover:text-green-900">Approve</button>
                                            <?php endif; ?>
                                            <?php if ($user['id'] != $_SESSION['id'] && ($user['role'] != 'SuperAdmin' || is_super_admin())): ?>
                                                 <button type="submit" name="delete_user" class="text-red-600 hover:text-red-900" onclick="return confirm('Are you sure you want to delete this user?');">Delete</button>
                                            <?php endif; ?>
                                        </form>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </main>
</div>
<?php include 'includes/footer.php'; ?>
