<?php
// FILE: admin/upload_knowledge.php

// --- Security & Session Management ---
// Corrected paths based on your file structure
require_once 'functions.php'; 
require_once 'db.php';   
require_once '../vendor/autoload.php'; // For PDF & XLSX parsing

// Ensure the user is logged in.
require_login();
// Ensure the user has the correct role to access this page.
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php"); // Redirect non-admins
    exit;
}

// --- CSRF Protection ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

$message = '';
$message_type = 'error';
$supportedTypes = ['pdf', 'docx', 'txt', 'csv', 'xlsx'];

// Helper: Split text into chunks (default ~1200 chars)
function split_into_chunks($text, $maxLength = 1200) {
    $chunks = [];
    $paragraphs = preg_split('/(\r?\n){2,}/', $text);
    $current = '';
    foreach ($paragraphs as $para) {
        if (strlen($current) + strlen($para) > $maxLength) {
            $chunks[] = trim($current);
            $current = '';
        }
        $current .= $para . "\n\n";
    }
    if (trim($current)) $chunks[] = trim($current);
    return $chunks;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['document'])) {
    // Verify CSRF Token
    if (!hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        die("CSRF token validation failed.");
    }

    $title = trim($_POST['title'] ?? 'Untitled Document');
    $docType = trim($_POST['doc_type'] ?? 'General');
    $file = $_FILES['document'];

    if ($file['error'] === UPLOAD_ERR_OK) {
        $uploadDir = '../uploads/knowledge_base/';
        if (!is_dir($uploadDir)) mkdir($uploadDir, 0755, true);

        $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, $supportedTypes)) {
            $message = "Error: Only PDF, DOCX, TXT, CSV, or XLSX files are allowed.";
        } else {
            $fileName = uniqid() . '-' . preg_replace("/[^a-zA-Z0-9.\-_]/", "", basename($file['name']));
            $filePath = $uploadDir . $fileName;

            if (move_uploaded_file($file['tmp_name'], $filePath)) {
                $contentText = '';
                try {
                    if ($ext === 'pdf') {
                        $parser = new \Smalot\PdfParser\Parser();
                        $pdf = $parser->parseFile($filePath);
                        $contentText = $pdf->getText();
                    } elseif ($ext === 'docx') {
                        $zip = new ZipArchive;
                        if ($zip->open($filePath) === true) {
                            $xml = $zip->getFromName('word/document.xml');
                            $zip->close();
                            $contentText = strip_tags(str_replace('</w:p>', "\n\n", $xml));
                        }
                    } elseif ($ext === 'txt' || $ext === 'csv') {
                         $contentText = file_get_contents($filePath);
                    } elseif ($ext === 'xlsx') {
                        $spreadsheet = \PhpOffice\PhpSpreadsheet\IOFactory::load($filePath);
                        $sheet = $spreadsheet->getActiveSheet();
                        $rows = [];
                        foreach ($sheet->toArray(null, true, true, true) as $row) {
                            $rows[] = implode(' | ', array_filter($row));
                        }
                        $contentText = implode("\n", $rows);
                    }
                } catch (Exception $e) {
                    $message = "Error extracting text: " . htmlspecialchars($e->getMessage());
                }

                if (empty(trim($contentText))) $contentText = "No extractable text found in this file.";
                
                $contentText = substr($contentText, 0, 50000);

                $stmt = $pdo->prepare(
                    "INSERT INTO knowledge_base (title, document_type, file_path, content_text, uploaded_by) VALUES (?, ?, ?, ?, ?)"
                );
                // Capture the logged-in user's ID from the session
                 $uploadedBy = $_SESSION['id']; 
                if ($stmt->execute([$title, $docType, $filePath, $contentText, $uploadedBy])) {
                    $docId = $pdo->lastInsertId();
                    $chunks = split_into_chunks($contentText, 1200);
                    $stmtChunk = $pdo->prepare(
                        "INSERT INTO knowledge_base_chunks (knowledge_base_id, chunk_index, content_text) VALUES (?, ?, ?)"
                    );
                    foreach ($chunks as $i => $chunk) {
                        $stmtChunk->execute([$docId, $i, $chunk]);
                    }
                    $message = "Document uploaded, text extracted, and chunked successfully!";
                    $message_type = 'success';
                } else {
                    $message = "Error: Could not save document to database.";
                }
            } else {
                $message = "Error: Could not move uploaded file.";
            }
        }
    } else {
        $message = "Error: File upload failed with code " . $file['error'];
    }
}

$docs = $pdo->query("SELECT id, title, document_type, uploaded_at, file_path FROM knowledge_base ORDER BY uploaded_at DESC")->fetchAll();

// Page specific variables for header
$pageTitle = "Admin | Knowledge Base";
?>
<?php require_once 'includes/header.php'; // Corrected path to header ?>

<body class="bg-gray-100 font-sans">
    <div class="container mx-auto p-4 sm:p-8">
        <div class="max-w-2xl mx-auto bg-white p-6 sm:p-8 rounded-2xl shadow-lg">
            <h1 class="text-3xl font-bold text-gray-800 mb-6">Upload to Knowledge Base</h1>
            <?php if ($message): ?>
                <div class="mb-6 p-4 rounded-lg text-sm <?= $message_type === 'error' ? 'bg-red-100 text-red-800' : 'bg-green-100 text-green-800' ?>">
                    <?= htmlspecialchars($message); ?>
                </div>
            <?php endif; ?>
            <form action="upload_knowledge.php" method="post" enctype="multipart/form-data" class="space-y-6">
                <!-- Add the CSRF token as a hidden field -->
                <input type="hidden" name="csrf_token" value="<?= htmlspecialchars($_SESSION['csrf_token']); ?>">
                
                <div>
                    <label for="title" class="block text-gray-700 font-semibold mb-2">Document Title</label>
                    <input type="text" id="title" name="title" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition" required>
                </div>
                <div>
                    <label for="doc_type" class="block text-gray-700 font-semibold mb-2">Document Type</label>
                    <input type="text" id="doc_type" name="doc_type" placeholder="e.g., CIDP, Budget Report" class="w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition">
                </div>
                <div>
                    <label for="document" class="block text-gray-700 font-semibold mb-2">Select File</label>
                    <input type="file" id="document" name="document" class="w-full text-gray-700 file:mr-4 file:py-2 file:px-4 file:rounded-lg file:border-0 file:text-sm file:font-semibold file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100" accept=".pdf,.docx,.txt,.csv,.xlsx" required>
                    <p class="text-xs text-gray-500 mt-1">Supported formats: PDF, DOCX, TXT, CSV, XLSX.</p>
                </div>
                <div>
                    <button type="submit" class="w-full bg-blue-600 text-white font-bold py-3 px-4 rounded-lg hover:bg-blue-700 transition-colors duration-300">Upload Document</button>
                </div>
            </form>
        </div>

        <div class="max-w-5xl mx-auto mt-12 bg-white p-6 sm:p-8 rounded-2xl shadow-lg">
            <h2 class="text-2xl font-semibold mb-6 text-gray-800">Knowledge Base Documents</h2>
            <div class="overflow-x-auto">
                <table class="w-full text-sm text-left text-gray-600">
                    <thead class="text-xs text-gray-700 uppercase bg-gray-100">
                        <tr>
                            <th class="p-3">Title</th>
                            <th class="p-3">Type</th>
                            <th class="p-3">Uploaded</th>
                            <th class="p-3 text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody>
                    <?php foreach ($docs as $doc): ?>
                        <tr class="bg-white border-b hover:bg-gray-50">
                            <td class="p-3 font-medium text-gray-900"><?= htmlspecialchars($doc['title']) ?></td>
                            <td class="p-3"><?= htmlspecialchars($doc['document_type']) ?></td>
                            <td class="p-3"><?= date("M j, Y", strtotime($doc['uploaded_at'])) ?></td>
                            <td class="p-3 text-center space-x-4">
                                <a href="../<?= htmlspecialchars($doc['file_path']) ?>" class="font-medium text-blue-600 hover:underline" target="_blank">Download</a>
                                <a href="view_knowledge.php?id=<?= $doc['id'] ?>" class="font-medium text-green-600 hover:underline">View Chunks</a>
                            </td>
                        </tr>
                    <?php endforeach ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</body>

<?php require_once 'includes/footer.php'; // Corrected path to footer ?>
</html>
