<?php
// FILE: admin/reset_password.php
// PURPOSE: Handles the actual password reset and notifies the user.

require_once 'functions.php';
require_once 'db.php';

$errors = [];
$success_message = '';
$token = $_GET['token'] ?? '';
$token_is_valid = false;
$user_email = '';
$user_phone = ''; // To store phone for SMS notification

if (empty($token)) {
    $errors['token'] = "No reset token provided. Please request a new link.";
} else {
    try {
        // Check if the token is valid and not expired, and get user contact info
        $stmt = $pdo->prepare("
            SELECT pr.email, u.phone_number 
            FROM password_resets pr 
            JOIN users u ON pr.email = u.email
            WHERE pr.token = :token AND pr.expires_at > NOW()
        ");
        $stmt->execute(['token' => $token]);
        $reset_request = $stmt->fetch();

        if ($reset_request) {
            $token_is_valid = true;
            $user_email = $reset_request['email'];
            $user_phone = $reset_request['phone_number'];
        } else {
            $errors['token'] = "This password reset link is invalid or has expired. Please request a new one.";
        }

    } catch (PDOException $e) {
        $errors['db'] = "A database error occurred.";
    }
}

// Handle form submission for new password
if ($_SERVER["REQUEST_METHOD"] == "POST" && $token_is_valid) {
    $password = trim($_POST['password']);
    $password_confirm = trim($_POST['password_confirm']);

    if (empty($password)) $errors['password'] = "Password is required.";
    if (strlen($password) < 8) $errors['password'] = "Password must be at least 8 characters long.";
    if ($password !== $password_confirm) $errors['password_confirm'] = "Passwords do not match.";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            // Hash the new password
            $hashed_password = password_hash($password, PASSWORD_DEFAULT);

            // Update the user's password in the users table
            $update_stmt = $pdo->prepare("UPDATE users SET password = :password WHERE email = :email");
            $update_stmt->execute(['password' => $hashed_password, 'email' => $user_email]);

            // Delete the used token from the password_resets table
            $delete_stmt = $pdo->prepare("DELETE FROM password_resets WHERE email = :email");
            $delete_stmt->execute(['email' => $user_email]);

            $pdo->commit();
            
            // Send confirmation notifications
            $email_subject = "Password Reset Confirmation";
            $email_body = "<p>This is a confirmation that the password for your Genowa Delivery Unit account has been successfully changed.</p><p>If you did not make this change, please contact an administrator immediately.</p>";
            send_email($user_email, $email_subject, $email_body);
            
            if ($user_phone) {
                $sms_message = "Your Genowa Delivery Unit password has been successfully reset. If you did not authorize this, please contact an admin.";
                send_sms($user_phone, $sms_message);
            }
            
            $success_message = "Your password has been reset successfully! You can now sign in with your new password.";
            $token_is_valid = false; // Prevent form from showing again

        } catch (PDOException $e) {
            $pdo->rollBack();
            $errors['db'] = "An error occurred while resetting your password. Please try again.";
        }
    }
}

include 'includes/header.php';
?>
<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        <div>
            <div class="text-center">
                 <h1 class="text-3xl font-extrabold text-brand-blue">GENOWA</h1>
                 <p class="text-lg font-semibold text-brand-orange tracking-wider">DELIVERY UNIT</p>
            </div>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Set New Password
            </h2>
        </div>
        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert">
                    <p><?php echo e($success_message); ?></p>
                    <a href="login.php" class="font-bold text-green-800 hover:underline">Click here to sign in</a>
                </div>
            <?php elseif (!$token_is_valid): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                     <?php foreach ($errors as $error): ?><p><?php echo e($error); ?></p><?php endforeach; ?>
                </div>
            <?php else: ?>
                <form class="space-y-6" action="reset_password.php?token=<?php echo e($token); ?>" method="post">
                     <div>
                        <label for="password" class="sr-only">New Password</label>
                        <input id="password" name="password" type="password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Enter new password (min. 8 characters)">
                        <?php if(isset($errors['password'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password']; ?></p><?php endif; ?>
                    </div>
                    <div>
                        <label for="password_confirm" class="sr-only">Confirm New Password</label>
                        <input id="password_confirm" name="password_confirm" type="password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Confirm new password">
                         <?php if(isset($errors['password_confirm'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password_confirm']; ?></p><?php endif; ?>
                    </div>
                    <div>
                        <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple">
                            Reset Password
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>
<?php include 'includes/footer.php'; ?>
