<?php
// FILE: admin/news_dashboard.php
// --- MODIFIED ---: Upgraded with status/category filters and improved UI for scheduled posts.

require_once 'functions.php';
require_once 'db.php';

// Security: Only Admins and SuperAdmins can access
require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- NEW: Filtering Logic ---
$filter_status = $_GET['status'] ?? '';
$filter_category = $_GET['category_id'] ?? '';
$where_clauses = [];
$params = [];

if (!empty($filter_status)) {
    $where_clauses[] = "a.status = :status";
    $params[':status'] = $filter_status;
}
if (!empty($filter_category)) {
    $where_clauses[] = "a.category_id = :category_id";
    $params[':category_id'] = $filter_category;
}

$where_sql = empty($where_clauses) ? '' : 'WHERE ' . implode(' AND ', $where_clauses);
$query_string = http_build_query(['status' => $filter_status, 'category_id' => $filter_category]);


// --- Setup Pagination ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 15;
$offset = ($page - 1) * $limit;

$articles = [];
$total_articles = 0;
$categories = []; // For filter dropdown

try {
    // --- NEW: Fetch categories for the filter dropdown ---
    $categories = $pdo->query("SELECT id, name FROM categories ORDER BY name ASC")->fetchAll();

    // Get total count for pagination (with filters)
    $count_sql = "SELECT COUNT(a.id) FROM news_articles a " . $where_sql;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_articles = $count_stmt->fetchColumn();
    $total_pages = ceil($total_articles / $limit);

    // Fetch articles for the current page (with filters)
    // --- MODIFIED: Query now joins categories and uses dynamic WHERE clause ---
    $sql = "SELECT a.*, u.name as author_name, c.name as category_name
            FROM news_articles a
            JOIN users u ON a.author_id = u.id
            LEFT JOIN categories c ON a.category_id = c.id
            $where_sql
            ORDER BY a.updated_at DESC
            LIMIT :limit OFFSET :offset";
            
    $stmt = $pdo->prepare($sql);
    // Bind filter params
    foreach ($params as $key => &$val) {
        $stmt->bindParam($key, $val);
    }
    // Bind pagination params
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    
    $stmt->execute();
    $articles = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching articles: " . $e->getMessage();
}

include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8 flex justify-between items-center">
        <div>
            <h1 class="text-lg font-semibold leading-6 text-gray-900">News & Updates</h1>
            <p class="mt-1 text-sm text-gray-500">Total articles: <?php echo e(number_format($total_articles)); ?></p>
        </div>
        <a href="manage_article.php" class="inline-flex items-center gap-x-2 rounded-md bg-brand-orange px-3.5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-orange-500">
            Write New Article
        </a>
    </div>
</header>

<!-- --- NEW: Filter Section --- -->
<div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 mt-6">
    <div class="bg-white p-4 rounded-lg shadow-sm">
        <form method="GET" action="news_dashboard.php" class="grid grid-cols-1 sm:grid-cols-3 md:grid-cols-4 gap-4 items-end">
            <div>
                <label for="status" class="block text-sm font-medium text-gray-700">Filter by Status</label>
                <select name="status" id="status" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm">
                    <option value="">All Statuses</option>
                    <option value="Published" <?php if ($filter_status == 'Published') echo 'selected'; ?>>Published</option>
                    <option value="Scheduled" <?php if ($filter_status == 'Scheduled') echo 'selected'; ?>>Scheduled</option>
                    <option value="Draft" <?php if ($filter_status == 'Draft') echo 'selected'; ?>>Draft</option>
                </select>
            </div>
            <div>
                <label for="category_id" class="block text-sm font-medium text-gray-700">Filter by Category</label>
                <select name="category_id" id="category_id" class="mt-1 block w-full rounded-md border-gray-300 shadow-sm">
                    <option value="">All Categories</option>
                    <?php foreach ($categories as $category): ?>
                        <option value="<?php echo $category['id']; ?>" <?php if ($filter_category == $category['id']) echo 'selected'; ?>>
                            <?php echo e($category['name']); ?>
                        </option>
                    <?php endforeach; ?>
                </select>
            </div>
            <div class="flex space-x-2">
                <button type="submit" class="inline-flex items-center justify-center rounded-md bg-brand-blue px-4 py-2 text-sm font-semibold text-white shadow-sm hover:bg-blue-800">Filter</button>
                <a href="news_dashboard.php" class="inline-flex items-center justify-center rounded-md bg-gray-200 px-4 py-2 text-sm font-semibold text-gray-700 shadow-sm hover:bg-gray-300">Reset</a>
            </div>
        </form>
    </div>
</div>
    
<div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 mt-6">
    <div class="overflow-x-auto bg-white rounded-lg shadow">
        <?php if (isset($db_error)): ?>
            <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
        <?php elseif (empty($articles)): ?>
            <div class="text-center p-12">
                <h3 class="mt-2 text-lg font-medium text-gray-900">No articles found.</h3>
                <p class="mt-1 text-sm text-gray-500">Try adjusting your filters or write a new article.</p>
            </div>
        <?php else: ?>
            <table class="min-w-full divide-y divide-gray-200">
                <thead class="bg-gray-50">
                    <tr>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title & Category</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Author</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                        <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Last Updated</th>
                        <th scope="col" class="relative px-6 py-3"><span class="sr-only">Actions</span></th>
                    </tr>
                </thead>
                <tbody class="bg-white divide-y divide-gray-200">
                    <?php foreach ($articles as $article): ?>
                        <tr>
                            <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                <div><?php echo e($article['title']); ?></div>
                                <div class="text-xs text-gray-500"><?php echo e($article['category_name'] ?? 'No Category'); ?></div>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo e($article['author_name']); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap">
                                <!-- --- MODIFIED: Status Badge Logic --- -->
                                <?php
                                    $status_classes = '';
                                    $status_text = e($article['status']);
                                    switch ($article['status']) {
                                        case 'Published':
                                            $status_classes = 'bg-green-100 text-green-800';
                                            break;
                                        case 'Scheduled':
                                            $status_classes = 'bg-blue-100 text-blue-800';
                                            $status_text .= ' for ' . date('M j, Y', strtotime($article['publish_at']));
                                            break;
                                        case 'Draft':
                                        default:
                                            $status_classes = 'bg-yellow-100 text-yellow-800';
                                            break;
                                    }
                                ?>
                                <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php echo $status_classes; ?>">
                                    <?php echo $status_text; ?>
                                </span>
                            </td>
                            <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date('M j, Y, g:ia', strtotime($article['updated_at'])); ?></td>
                            <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium space-x-4">
                                <!-- --- NEW: View Link --- -->
                                <?php if ($article['status'] == 'Published' || ($article['status'] == 'Scheduled' && strtotime($article['publish_at']) <= time())): ?>
                                    <a href="../article.php?slug=<?php echo e($article['slug']); ?>" target="_blank" class="text-green-600 hover:text-green-900">View</a>
                                <?php endif; ?>
                                <a href="manage_article.php?id=<?php echo $article['id']; ?>" class="text-brand-purple hover:text-brand-orange">Edit</a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
</div>

<!-- --- MODIFIED: Pagination links now include filter parameters --- -->
<div class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 mt-6 flex justify-between items-center">
    <?php if ($total_pages > 1): ?>
        <div>
            <?php if ($page > 1): ?>
                <a href="?page=<?php echo $page - 1; ?>&<?php echo $query_string; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
            <?php endif; ?>
        </div>
        <div class="text-sm text-gray-700">Page <?php echo $page; ?> of <?php echo $total_pages; ?></div>
        <div>
            <?php if ($page < $total_pages): ?>
                <a href="?page=<?php echo $page + 1; ?>&<?php echo $query_string; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
            <?php endif; ?>
        </div>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>