<?php
// FILE: admin/manage_gallery.php
// PURPOSE: To upload, view, and manage images within a specific gallery.

require_once 'functions.php';
require_once 'db.php';

require_login();

$gallery_id = $_GET['id'] ?? null;
if (!$gallery_id) {
    header("Location: gallery_dashboard.php");
    exit;
}

$gallery = null;
$images = [];
$feedback = [];

try {
    // Fetch gallery details
    $gallery_stmt = $pdo->prepare("SELECT * FROM galleries WHERE id = :id");
    $gallery_stmt->execute(['id' => $gallery_id]);
    $gallery = $gallery_stmt->fetch();

    if (!$gallery) {
        header("Location: gallery_dashboard.php");
        exit;
    }
} catch (PDOException $e) {
    die("Error fetching gallery details: " . $e->getMessage());
}


// --- Handle Multiple File Uploads ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_FILES['image_uploads'])) {
    $upload_dir = 'uploads/';
    $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
    $success_count = 0;
    $error_count = 0;

    foreach ($_FILES['image_uploads']['name'] as $key => $name) {
        if ($_FILES['image_uploads']['error'][$key] === UPLOAD_ERR_OK) {
            $tmp_name = $_FILES['image_uploads']['tmp_name'][$key];
            $file_type = $_FILES['image_uploads']['type'][$key];
            $file_size = $_FILES['image_uploads']['size'][$key];

            if (in_array($file_type, $allowed_types) && $file_size <= 5000000) {
                $file_name = 'media_' . uniqid() . '.' . strtolower(pathinfo($name, PATHINFO_EXTENSION));
                $destination = $upload_dir . $file_name;

                if (move_uploaded_file($tmp_name, $destination)) {
                    try {
                        $sql = "INSERT INTO media_gallery (file_name, file_type, file_size, alt_text, uploaded_by, gallery_id) VALUES (:file_name, :file_type, :file_size, :alt_text, :uploaded_by, :gallery_id)";
                        $stmt = $pdo->prepare($sql);
                        $stmt->execute([
                            'file_name' => $file_name, 'file_type' => $file_type,
                            'file_size' => $file_size, 'alt_text' => pathinfo($name, PATHINFO_FILENAME),
                            'uploaded_by' => $_SESSION['id'], 'gallery_id' => $gallery_id
                        ]);
                        $success_count++;
                    } catch (PDOException $e) {
                        $error_count++; unlink($destination);
                    }
                } else { $error_count++; }
            } else { $error_count++; }
        }
    }
    
    if ($success_count > 0) { log_action($pdo, "Uploaded {$success_count} images to gallery", "Gallery", $gallery_id); }
    $feedback['message'] = "Upload complete. {$success_count} files uploaded successfully.";
    if ($error_count > 0) { $feedback['message'] .= " {$error_count} files failed due to size or type errors."; }
    $feedback['type'] = 'success';
}


// --- Setup Pagination for images in this gallery ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 24; 
$offset = ($page - 1) * $limit;

try {
    $count_stmt = $pdo->prepare("SELECT COUNT(id) FROM media_gallery WHERE gallery_id = :gallery_id");
    $count_stmt->execute(['gallery_id' => $gallery_id]);
    $total_images = $count_stmt->fetchColumn();
    $total_pages = ceil($total_images / $limit);

    $sql = "SELECT * FROM media_gallery WHERE gallery_id = :gallery_id ORDER BY uploaded_at DESC LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':gallery_id', $gallery_id, PDO::PARAM_INT);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $images = $stmt->fetchAll();
} catch (PDOException $e) {
    $db_error = "Error fetching media: " . $e->getMessage();
}


include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">Manage Gallery: <?php echo e($gallery['title']); ?></h1>
        <p class="mt-1 text-sm text-gray-500"><?php echo e($gallery['description']); ?></p>
    </div>
</header>
    
<div class="p-4 bg-white rounded-lg shadow mt-6">
    <form action="manage_gallery.php?id=<?php echo e($gallery['id']); ?>" method="post" enctype="multipart/form-data" id="upload-form">
        <div id="drop-area" class="relative block w-full rounded-lg border-2 border-dashed border-gray-300 p-12 text-center hover:border-gray-400 focus:outline-none focus:ring-2 focus:ring-brand-purple focus:ring-offset-2">
            <svg class="mx-auto h-12 w-12 text-gray-400" fill="none" viewBox="0 0 24 24" stroke-width="1.5" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" d="M3 16.5v2.25A2.25 2.25 0 005.25 21h13.5A2.25 2.25 0 0021 18.75V16.5m-13.5-9L12 3m0 0l4.5 4.5M12 3v13.5" /></svg>
            <span class="mt-2 block text-sm font-semibold text-gray-900">Drag & drop files here or click to upload</span>
            <input type="file" name="image_uploads[]" id="image_uploads" multiple class="absolute inset-0 w-full h-full opacity-0 cursor-pointer">
        </div>
        <div id="file-list" class="mt-4 text-sm text-gray-500"></div>
        <div class="mt-4 flex justify-end">
            <button type="submit" class="rounded-md bg-brand-orange px-3 py-2 text-sm font-semibold text-white shadow-sm hover:bg-opacity-90">Upload Files</button>
        </div>
    </form>
</div>

<div class="mt-6">
     <?php if (!empty($feedback)): ?>
        <div class="mb-4 p-4 rounded-md <?php echo $feedback['type'] === 'success' ? 'bg-green-100 text-green-700' : 'bg-red-100 text-red-700'; ?>">
            <?php echo e($feedback['message']); ?>
        </div>
    <?php endif; ?>
    <div class="grid grid-cols-2 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8 gap-4">
        <?php foreach($images as $image): ?>
            <div class="relative group">
                <img src="uploads/<?php echo e($image['file_name']); ?>" alt="<?php echo e($image['alt_text']); ?>" class="w-full h-32 object-cover rounded-lg shadow-md">
                 <div class="absolute inset-0 bg-black/50 opacity-0 group-hover:opacity-100 transition-opacity flex items-center justify-center">
                     <form action="delete_media.php" method="post" onsubmit="return confirm('Are you sure you want to permanently delete this image?');">
                        <input type="hidden" name="image_id" value="<?php echo e($image['id']); ?>">
                        <input type="hidden" name="gallery_id" value="<?php echo e($gallery['id']); ?>">
                        <button type="submit" name="delete_image" class="text-white text-xs font-bold bg-red-600/70 p-2 rounded-full">Delete</button>
                    </form>
                </div>
            </div>
        <?php endforeach; ?>
    </div>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?id=<?php echo e($gallery_id); ?>&page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    <div class="text-sm text-gray-700">Page <?php echo $page; ?> of <?php echo $total_pages; ?></div>
    <?php if ($page < $total_pages): ?>
        <a href="?id=<?php echo e($gallery_id); ?>&page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const dropArea = document.getElementById('drop-area');
    const fileInput = document.getElementById('image_uploads');
    const fileList = document.getElementById('file-list');

    dropArea.addEventListener('dragover', (e) => {
        e.preventDefault();
        dropArea.classList.add('border-brand-purple');
    });
    dropArea.addEventListener('dragleave', () => {
        dropArea.classList.remove('border-brand-purple');
    });
    dropArea.addEventListener('drop', (e) => {
        e.preventDefault();
        dropArea.classList.remove('border-brand-purple');
        fileInput.files = e.dataTransfer.files;
        updateFileList();
    });
    fileInput.addEventListener('change', updateFileList);

    function updateFileList() {
        fileList.innerHTML = '';
        if (fileInput.files.length > 0) {
            const names = Array.from(fileInput.files).map(f => f.name).join(', ');
            fileList.textContent = `Selected: ${names}`;
        }
    }
});
</script>

<?php include 'includes/footer.php'; ?>
