<?php
// FILE: admin/manage_article.php
// PURPOSE: A form for creating and editing news articles with a rich text editor and advanced publishing workflow.

require_once 'functions.php';
require_once 'db.php';

require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- CSRF Protection ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// --- Initialize variables ---
$article = [
    'id' => null, 'title' => '', 'slug' => '', 'content' => '',
    'featured_image' => null, 'project_id' => null, 'category_id' => null,
    'author_name' => '', // For display
];
$article_tags = [];
$errors = [];
$page_title = "Write New Article";
$is_edit_mode = false;

// --- Fetch data for form dropdowns (Projects, Categories, Tags) ---
try {
    $approved_projects = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC")->fetchAll();
    $categories = $pdo->query("SELECT id, name FROM categories ORDER BY name ASC")->fetchAll();
    $all_tags = $pdo->query("SELECT id, name FROM tags ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db_load'] = "Could not fetch page data.";
}

// --- Handle GET request (edit mode) ---
if (isset($_GET['id']) && !empty(trim($_GET['id']))) {
    $is_edit_mode = true;
    $article_id = trim($_GET['id']);
    $page_title = "Edit Article";

    // Join with users table to get author's name
    $stmt = $pdo->prepare("
        SELECT a.*, u.name AS author_name 
        FROM news_articles a
        LEFT JOIN users u ON a.author_id = u.id
        WHERE a.id = :id
    ");
    $stmt->execute(['id' => $article_id]);
    $article = $stmt->fetch();

    if (!$article) {
        header("location: news_dashboard.php");
        exit;
    }

    // Fetch existing tags for the article
    $tags_stmt = $pdo->prepare("SELECT tag_id FROM article_tags WHERE article_id = :article_id");
    $tags_stmt->execute(['article_id' => $article_id]);
    $article_tags = $tags_stmt->fetchAll(PDO::FETCH_COLUMN);
}

// --- Handle POST request ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // CSRF Token Validation
    if (!isset($_POST['csrf_token']) || !hash_equals($_SESSION['csrf_token'], $_POST['csrf_token'])) {
        $errors['csrf'] = 'Invalid request. Please try again.';
    } else {
        $article_id = $_POST['id'] ?? null;
        $article_data = [
            'title' => trim($_POST['title']),
            'content' => trim($_POST['content']),
            'status' => $_POST['status'] ?? 'Draft',
            'project_id' => empty($_POST['project_id']) ? null : $_POST['project_id'],
            'category_id' => empty($_POST['category_id']) ? null : $_POST['category_id'],
            'publish_at' => empty($_POST['publish_at']) ? null : $_POST['publish_at'],
            'slug' => $_POST['slug'] ?? create_slug($_POST['title']),
        ];
        $posted_tags = isset($_POST['tags']) && is_array($_POST['tags']) ? $_POST['tags'] : [];

        // Validation
        if (empty($article_data['title'])) $errors['title'] = "Article title is required.";
        if (empty($article_data['project_id'])) $errors['project_id'] = "You must link this article to a project.";
        if (empty($article_data['category_id'])) $errors['category_id'] = "You must select a category.";
        if ($article_data['status'] === 'Scheduled' && empty($article_data['publish_at'])) {
            $errors['publish_at'] = 'A publication date is required for scheduled articles.';
        }

        // Image Upload Validation
        $new_image_filename = $_POST['existing_image'] ?? null;
        if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] == UPLOAD_ERR_OK) {
            // ... (strict file upload validation logic from previous steps) ...
             $new_image_filename = '...'; // a securely generated filename
        }
        $article_data['featured_image'] = $new_image_filename;

        // Save to DB if no errors
        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                if ($article_id) {
                    $sql = "UPDATE news_articles SET title = :title, slug = :slug, content = :content, status = :status, project_id = :project_id, category_id = :category_id, publish_at = :publish_at, featured_image = :featured_image WHERE id = :id";
                } else {
                    $sql = "INSERT INTO news_articles (title, slug, content, status, project_id, category_id, publish_at, featured_image, author_id) VALUES (:title, :slug, :content, :status, :project_id, :category_id, :publish_at, :featured_image, :author_id)";
                }

                $stmt = $pdo->prepare($sql);
                $params = [
                    'title' => $article_data['title'],
                    'slug' => $article_data['slug'],
                    'content' => $article_data['content'],
                    'status' => $article_data['status'],
                    'project_id' => $article_data['project_id'],
                    'category_id' => $article_data['category_id'],
                    'publish_at' => $article_data['publish_at'],
                    'featured_image' => $article_data['featured_image'],
                ];
                
                if ($article_id) {
                    $params['id'] = $article_id;
                } else {
                    $params['author_id'] = $_SESSION['id'];
                }

                $stmt->execute($params);
                $current_article_id = $article_id ?: $pdo->lastInsertId();

                // Handle Tags
                $delete_tags_stmt = $pdo->prepare("DELETE FROM article_tags WHERE article_id = :article_id");
                $delete_tags_stmt->execute(['article_id' => $current_article_id]);
                if (!empty($posted_tags)) {
                    $insert_tag_stmt = $pdo->prepare("INSERT INTO article_tags (article_id, tag_id) VALUES (:article_id, :tag_id)");
                    foreach ($posted_tags as $tag_id) {
                        $insert_tag_stmt->execute(['article_id' => $current_article_id, 'tag_id' => $tag_id]);
                    }
                }
                
                $pdo->commit();

                header("location: news_dashboard.php?success=saved");
                exit;

            } catch (PDOException $e) {
                $pdo->rollBack();
                error_log("Database Error in manage_article.php: " . $e->getMessage());
                $errors['db'] = "A database error occurred.";
            }
        }
    }
}

include 'includes/header.php';
?>
<link href="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/css/tom-select.bootstrap5.min.css" rel="stylesheet">
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<style>
    .ts-control {
        padding: 0.5rem 0.75rem;
        border-radius: 0.375rem;
        border: 1px solid #d1d5db;
    }

    /* --- NEW: Style for Tom Select Dropdown Panel --- */
    .ts-dropdown {
        background-color: #fff; /* Solid white background */
        z-index: 20; /* Ensure it's above other page content */
        border: 1px solid #d1d5db; /* Consistent border */
        border-radius: 0.375rem; /* Consistent corner radius */
        /* Optional: Add a subtle shadow for depth */
        box-shadow: 0 4px 6px -1px rgb(0 0 0 / 0.1), 0 2px 4px -2px rgb(0 0 0 / 0.1);
    }
</style>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900"><?php echo e($page_title); ?></h1>
    </div>
</header>

<form id="article-form" method="post" enctype="multipart/form-data">
    <input type="hidden" name="id" value="<?php echo e($article['id']); ?>">
    <input type="hidden" name="content" id="quill-content">
    <input type="hidden" name="csrf_token" value="<?php echo e($_SESSION['csrf_token']); ?>">

    <div class="mx-auto max-w-7xl py-6 grid grid-cols-1 lg:grid-cols-3 gap-6">
        <div class="lg:col-span-2 space-y-6">
            <div class="bg-white p-6 rounded-lg shadow">
                <label for="title" class="block text-sm font-medium text-gray-700">Article Title</label>
                <input type="text" name="title" id="title" value="<?php echo e($article['title']); ?>" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm px-4 py-3 text-lg">
                <div class="mt-2">
                    <label for="slug" class="block text-xs font-medium text-gray-500">URL Slug</label>
                    <input type="text" name="slug" id="slug" value="<?php echo e($article['slug']); ?>" class="mt-1 block w-full rounded-md border-gray-300 text-sm text-gray-600">
                </div>
            </div>

            <div class="bg-white rounded-lg shadow p-6">
                <label class="block text-sm font-medium text-gray-700 mb-2">Article Content</label>
                <div id="editor" class="h-96"><?php echo $article['content'] ?? ''; ?></div>
            </div>
        </div>

        <div class="space-y-6">
            <div class="bg-white p-6 rounded-lg shadow">
                 <div class="mb-4">
                    <label for="category_id" class="block text-sm font-medium text-gray-700">Category</label>
                    <select id="category_id" name="category_id" required>
                        <option value="">Select a category...</option>
                        <?php foreach ($categories as $cat): ?>
                            <option value="<?php echo $cat['id']; ?>" <?php if (($article['category_id'] ?? '') == $cat['id']) echo 'selected'; ?>><?php echo e($cat['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="mb-4">
                    <label for="tags" class="block text-sm font-medium text-gray-700">Tags</label>
                    <select id="tags" name="tags[]" multiple>
                        <?php foreach ($all_tags as $tag): ?>
                           <option value="<?php echo $tag['id']; ?>" <?php if (in_array($tag['id'], $article_tags)) echo 'selected'; ?>><?php echo e($tag['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div>
                    <label for="project_id" class="block text-sm font-medium text-gray-700">Link to Project</label>
                    <select id="project_id" name="project_id" required>
                        <option value="">Select a project...</option>
                        <?php foreach ($approved_projects as $proj): ?>
                            <option value="<?php echo $proj['id']; ?>" <?php if (($article['project_id'] ?? '') == $proj['id']) echo 'selected'; ?>><?php echo e($proj['name']); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <?php if ($is_edit_mode && !empty($article['author_name'])): ?>
                <div class="mt-4">
                    <label class="block text-sm font-medium text-gray-700">Author</label>
                    <p class="mt-1 text-md text-gray-800"><?php echo e($article['author_name']); ?></p>
                </div>
                <?php endif; ?>
            </div>

            <div class="bg-white p-6 rounded-lg shadow">
                <label for="featured_image" class="block text-sm font-medium text-gray-700">Featured Image</label>
                <input type="hidden" name="existing_image" value="<?php echo e($article['featured_image']); ?>">
                <img id="image-preview" src="<?php echo (!empty($article['featured_image']) && file_exists(__DIR__ . '/uploads/' . $article['featured_image'])) ? 'uploads/' . e($article['featured_image']) : ''; ?>" 
                     class="w-full h-auto rounded mt-2 mb-4 <?php echo empty($article['featured_image']) ? 'hidden' : ''; ?>" alt="Image Preview">
                <input type="file" name="featured_image" id="featured_image" class="block w-full text-sm mt-2" accept="image/*">
            </div>

            <div class="text-right">
                <a href="news_dashboard.php" class="text-sm text-gray-500 hover:underline mr-4">Cancel</a>
                <button type="button" id="open-publish-panel" class="bg-brand-orange text-white px-4 py-2 rounded hover:bg-opacity-90">
                    Review & Publish
                </button>
            </div>
        </div>
    </div>

    <div id="publish-panel" class="fixed inset-0 bg-gray-800 bg-opacity-50 hidden flex justify-end z-50">
        <div class="bg-white w-full max-w-md shadow-xl p-6 space-y-6 overflow-y-auto">
            <div class="flex justify-between items-center">
                <h3 class="text-xl font-semibold text-gray-900">Publish Article</h3>
                <button type="button" id="close-publish-panel" class="text-gray-500 hover:text-gray-800 text-2xl">&times;</button>
            </div>
            <div>
                <h4 class="text-lg font-medium text-gray-800">Preview</h4>
                <div class="mt-2 border rounded-lg p-4">
                    <img id="panel-image-preview" src="" class="w-full h-auto rounded-md mb-3" alt="Image Preview" style="display:none;">
                    <h5 id="panel-title-preview" class="text-xl font-bold"></h5>
                </div>
            </div>
            <div>
                <h4 class="text-lg font-medium text-gray-800">Publishing Options</h4>
                <div class="mt-2 space-y-3">
                    <label class="flex items-center p-3 border rounded-lg cursor-pointer has-[:checked]:bg-blue-50 has-[:checked]:border-blue-400">
                        <input type="radio" name="publish_option" value="now" class="h-4 w-4" checked>
                        <span class="ml-3 text-md">Publish Now</span>
                    </label>
                    <label class="flex items-center p-3 border rounded-lg cursor-pointer has-[:checked]:bg-blue-50 has-[:checked]:border-blue-400">
                        <input type="radio" name="publish_option" value="schedule" class="h-4 w-4">
                        <span class="ml-3 text-md">Schedule for later</span>
                    </label>
                </div>
            </div>
            <div id="panel-schedule-wrapper" class="hidden">
                <label for="panel_publish_at" class="block text-sm font-medium text-gray-700">Publication Date</label>
                <input type="datetime-local" id="panel_publish_at" class="mt-1 block w-full rounded-md border-gray-300">
            </div>
            <div class="pt-4 border-t">
                <button type="submit" id="final-publish-button" class="w-full bg-green-600 text-white px-4 py-3 rounded-lg hover:bg-green-700 text-lg font-bold">
                    Confirm & Publish
                </button>
            </div>
        </div>
    </div>
</form>

<script src="https://cdn.jsdelivr.net/npm/tom-select@2.3.1/dist/js/tom-select.complete.min.js"></script>
<script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    // --- Setup ---
    var quill = new Quill('#editor', {
        theme: 'snow',
        modules: { toolbar: [['bold', 'italic', 'underline'], [{ 'header': [1, 2, false] }], ['link', 'blockquote'], [{ 'list': 'ordered'}, { 'list': 'bullet' }], ['clean']] }
    });
    new TomSelect('#project_id');
    new TomSelect('#category_id');
    new TomSelect('#tags', { plugins: ['remove_button'], create: false });

    // --- Main Form & Publish Panel Elements ---
    const articleForm = document.getElementById('article-form');
    const openPublishBtn = document.getElementById('open-publish-panel');
    const closePublishBtn = document.getElementById('close-publish-panel');
    const publishPanel = document.getElementById('publish-panel');
    const finalPublishBtn = document.getElementById('final-publish-button');
    const panelImagePreview = document.getElementById('panel-image-preview');
    const panelTitlePreview = document.getElementById('panel-title-preview');
    const scheduleWrapper = document.getElementById('panel-schedule-wrapper');
    const panelPublishAt = document.getElementById('panel_publish_at');
    const publishOptions = document.querySelectorAll('input[name="publish_option"]');

    // --- Add hidden inputs to the main form to hold the final status and date ---
    let statusInput = document.createElement('input');
    statusInput.type = 'hidden';
    statusInput.name = 'status';
    articleForm.appendChild(statusInput);
    
    let publishAtInput = document.createElement('input');
    publishAtInput.type = 'hidden';
    publishAtInput.name = 'publish_at';
    articleForm.appendChild(publishAtInput);

    // --- Event Listeners ---
    openPublishBtn.addEventListener('click', function() {
        panelTitlePreview.textContent = document.getElementById('title').value || 'Untitled Article';
        const mainImageSrc = document.getElementById('image-preview').src;
        if (mainImageSrc && !mainImageSrc.endsWith('/')) {
            panelImagePreview.src = mainImageSrc;
            panelImagePreview.style.display = 'block';
        } else {
            panelImagePreview.style.display = 'none';
        }
        publishPanel.classList.remove('hidden');
    });

    closePublishBtn.addEventListener('click', () => publishPanel.classList.add('hidden'));

    publishOptions.forEach(radio => {
        radio.addEventListener('change', function() {
            if (this.value === 'schedule') {
                scheduleWrapper.classList.remove('hidden');
                finalPublishBtn.textContent = 'Confirm & Schedule';
            } else {
                scheduleWrapper.classList.add('hidden');
                finalPublishBtn.textContent = 'Confirm & Publish';
            }
        });
    });

    articleForm.addEventListener('submit', function() {
        const selectedOption = document.querySelector('input[name="publish_option"]:checked').value;
        if (selectedOption === 'now') {
            statusInput.value = 'Published';
            publishAtInput.value = '';
        } else {
            statusInput.value = 'Scheduled';
            publishAtInput.value = panelPublishAt.value;
        }
        document.getElementById('quill-content').value = quill.root.innerHTML;
    });

    // --- Live Image Preview ---
    const imageInput = document.getElementById('featured_image');
    const imagePreview = document.getElementById('image-preview');
    imageInput.addEventListener('change', function(e) {
        const file = e.target.files[0];
        if (file) {
            const reader = new FileReader();
            reader.onload = (event) => {
                imagePreview.src = event.target.result;
                imagePreview.classList.remove('hidden');
            }
            reader.readAsDataURL(file);
        }
    });

    // --- Live Slug Generation ---
    const titleInput = document.getElementById('title');
    const slugInput = document.getElementById('slug');
    const createSlug = (text) => text.toLowerCase().trim().replace(/\s+/g, '-').replace(/[^\w\-]+/g, '').replace(/\-\-+/g, '-').replace(/^-+/, '').replace(/-+$/, '');
    titleInput.addEventListener('keyup', () => { slugInput.value = createSlug(titleInput.value); });
});
</script>

<?php include 'includes/footer.php'; ?>