<?php
// FILE: admin/login.php
// PURPOSE: The main login page for the admin panel.

// No need to start session here, header.php does it.
// We include the header later to avoid output before header() redirect.

$error_message = '';

// Check if form is submitted
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    
    // Require DB connection only when form is submitted
    require_once 'db.php';

    $email = trim($_POST['email']);
    $password = trim($_POST['password']);

    if (empty($email) || empty($password)) {
        $error_message = "Please enter both email and password.";
    } else {
        try {
            // UPDATED: Fetch the new is_active column
            $stmt = $pdo->prepare("SELECT id, name, email, password, role, ward_id, is_active FROM users WHERE email = :email LIMIT 1");
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch();

            // Verify the user exists and the password is correct
            if ($user && password_verify($password, $user['password'])) {
                
                // NEW: Check if the user's account is active
                if ($user['is_active'] != 1) {
                    $error_message = "Your account is pending approval. Please contact an administrator.";
                } else {
                    // Password is correct and account is active, start a new session
                    session_start();
                    session_regenerate_id(true); // Prevents session fixation

                    // Store data in session variables
                    $_SESSION["loggedin"] = true;
                    $_SESSION["id"] = $user['id'];
                    $_SESSION["name"] = $user['name'];
                    $_SESSION["role"] = $user['role'];
                    $_SESSION["ward_id"] = $user['ward_id'];

                    // Redirect user to the dashboard page
                    header("location: dashboard.php");
                    exit; // Important to call exit after a header redirect
                }

            } else {
                // Display an error message if email doesn't exist or password is not valid.
                $error_message = "The email or password you entered is incorrect.";
            }
        } catch (PDOException $e) {
            // In a real application, you should log this error to a file.
            $error_message = "A database error occurred. Please try again later.";
            // For debugging: error_log("Login Error: " . $e->getMessage()); 
        }
    }
}

// Now include the header, as we are done with any potential redirects.
include 'includes/header.php';
?>

<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        
        <div>
            <!-- UPDATED STANDARD LOGO -->
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Admin Panel Sign In
            </h2>
        </div>

        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if (!empty($error_message)): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                    <p><?php echo htmlspecialchars($error_message); ?></p>
                </div>
            <?php endif; ?>
    
            <form class="space-y-6" action="<?php echo htmlspecialchars($_SERVER["PHP_SELF"]); ?>" method="post">
                <div>
                    <label for="email-address" class="sr-only">Email address</label>
                    <input id="email-address" name="email" type="email" autocomplete="email" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Email address">
                </div>
                <div>
                    <label for="password" class="sr-only">Password</label>
                    <input id="password" name="password" type="password" autocomplete="current-password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Password">
                </div>
    
                <div class="flex items-center justify-between">
                    <div class="flex items-center">
                        <input id="remember-me" name="remember-me" type="checkbox" class="h-4 w-4 text-brand-orange focus:ring-brand-purple border-gray-300 rounded">
                        <label for="remember-me" class="ml-2 block text-sm text-gray-900">
                            Remember me
                        </label>
                    </div>
    
                    <div class="text-sm">
                        <a href="forgot_password.php" class="font-medium text-brand-purple hover:text-brand-orange">
                            Forgot your password?
                        </a>
                    </div>
                </div>
    
                <div>
                    <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out">
                        Sign in
                    </button>
                </div>
            </form>
            <p class="text-center text-sm text-gray-600">
                Don't have an account? 
                <a href="signup.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign up here
                </a>
            </p>
        </div>
    </div>
</div>

<?php
// Include the footer template
include 'includes/footer.php';
?>
