<?php
// FILE: functions.php
// PURPOSE: Unified, robust functions library for the admin panel and site-wide use.

date_default_timezone_set('Africa/Nairobi');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

/**
 * Notifies SuperAdmins if pending projects or users pass thresholds, with a cooldown.
 */
function check_and_notify_pending_items(PDO $pdo): void
{
    $notification_cooldown = 3600 * 6; // 6 hours
    $notification_dir = __DIR__ . '/notifications/';
    if (!is_dir($notification_dir)) {
        mkdir($notification_dir, 0755, true);
    }
    $last_project_notification_file = $notification_dir . 'last_project_notification.txt';
    $last_user_notification_file = $notification_dir . 'last_user_notification.txt';

    try {
        // 1. Check pending projects
        $project_count_stmt = $pdo->query("SELECT COUNT(id) FROM projects WHERE is_approved = 0");
        $pending_projects_count = $project_count_stmt->fetchColumn();
        if ($pending_projects_count > 10) {
            $last_sent_time = file_exists($last_project_notification_file) ? filemtime($last_project_notification_file) : 0;
            if (time() - $last_sent_time > $notification_cooldown) {
                $super_admins_stmt = $pdo->query("SELECT email, phone_number FROM users WHERE role = 'SuperAdmin' AND is_active = 1");
                $super_admins = $super_admins_stmt->fetchAll();
                $message = "Genowa Alert: There are $pending_projects_count projects pending approval. Please log in to the admin panel to review them.";
                $subject = "Alert: High Number of Pending Projects";
                foreach ($super_admins as $admin) {
                    if (!empty($admin['email'])) {
                        send_email($admin['email'], $subject, "<p>$message</p>");
                    }
                    if (!empty($admin['phone_number'])) {
                        send_sms($admin['phone_number'], $message);
                    }
                }
                touch($last_project_notification_file);
            }
        }
        // 2. Check pending users
        $user_count_stmt = $pdo->query("SELECT COUNT(id) FROM users WHERE is_active = 0");
        $pending_users_count = $user_count_stmt->fetchColumn();
        if ($pending_users_count > 3) {
            $last_sent_time = file_exists($last_user_notification_file) ? filemtime($last_user_notification_file) : 0;
            if (time() - $last_sent_time > $notification_cooldown) {
                $super_admins_stmt = $pdo->query("SELECT email, phone_number FROM users WHERE role = 'SuperAdmin' AND is_active = 1");
                $super_admins = $super_admins_stmt->fetchAll();
                $message = "Genowa Alert: There are $pending_users_count new users awaiting approval. Please log in to the User Management page to review.";
                $subject = "Alert: New Users Awaiting Approval";
                foreach ($super_admins as $admin) {
                    if (!empty($admin['email'])) {
                        send_email($admin['email'], $subject, "<p>$message</p>");
                    }
                    if (!empty($admin['phone_number'])) {
                        send_sms($admin['phone_number'], $message);
                    }
                }
                touch($last_user_notification_file);
            }
        }
    } catch (PDOException $e) {
        error_log("Notification Check Error: " . $e->getMessage());
    }
}

/**
 * Sends an email (basic).
 */
function send_email(string $to, string $subject, string $body): bool
{
    $headers = "MIME-Version: 1.0\r\n";
    $headers .= "Content-type:text/html;charset=UTF-8\r\n";
    $headers .= 'From: Genowa Delivery Unit <hello@genowa.co.ke>' . "\r\n";
    return mail($to, $subject, $body, $headers);
}

/**
 * Sends an SMS using Onfon Media API.
 */
function send_sms(string $phoneNumber, string $message): bool
{
    // Sanitize phone number to international format
    if (substr($phoneNumber, 0, 1) == '+') { $phoneNumber = substr($phoneNumber, 1); }
    if (substr($phoneNumber, 0, 1) == '0') { $phoneNumber = '254' . substr($phoneNumber, 1); }
    if (substr($phoneNumber, 0, 3) != '254') { return false; }
    $api_url = 'https://api.onfonmedia.co.ke/v1/sms/SendBulkSMS';
    $api_key = 'HDv7xtZaIBLJwVlbkfYj65z2pRQGsOoPqihgy8n140AF39Km'; // Example API Key
    $client_id = 'Agwesh'; // 
    $sender_id = 'GENOWA';
    $payload = [
        'SenderId' => $sender_id,
        'MessageParameters' => [['Number' => $phoneNumber, 'Text' => $message]],
        'ApiKey' => $api_key,
        'ClientId' => $client_id
    ];
    $ch = curl_init($api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json', 'AccessKey: ' . $api_key]);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    if ($http_code == 200) {
        $result = json_decode($response, true);
        return isset($result['Data'][0]['ResultCode']) && $result['Data'][0]['ResultCode'] == 0;
    }
    return false;
}

// ----------- USER AUTH / ROLES (UNIFIED NAMING) -----------

function require_login(): void {
    if (session_status() === PHP_SESSION_NONE) { session_start(); }
    if (!isset($_SESSION["loggedin"]) || $_SESSION["loggedin"] !== true) {
        header("location: login.php"); exit;
    }
}
function is_super_admin(): bool {
    return isset($_SESSION["role"]) && $_SESSION["role"] === 'SuperAdmin';
}
function is_admin(): bool {
    return isset($_SESSION["role"]) && in_array($_SESSION["role"], ['Admin', 'SuperAdmin']);
}
function is_approver(): bool {
    return isset($_SESSION["role"]) && $_SESSION["role"] === 'Approver';
}
function is_inputer(): bool {
    return isset($_SESSION["role"]) && $_SESSION["role"] === 'Inputer';
}
function require_super_admin(): void {
    if (!is_super_admin()) { header("location: dashboard.php"); exit; }
}

// ----------- GENERAL HELPERS -----------

/**
 * Escapes output for HTML context.
 */
function e(?string $string): string {
    return htmlspecialchars($string ?? '', ENT_QUOTES, 'UTF-8');
}

// ----------- AUDIT & LOGGING -----------

/**
 * Logs user actions to audit_trails table.
 */
function log_action(PDO $pdo, string $action, string $target_type, int $target_id): void {
    try {
        $sql = "INSERT INTO audit_trails (user_id, user_name, action, target_type, target_id)
                VALUES (:user_id, :user_name, :action, :target_type, :target_id)";
        $stmt = $pdo->prepare($sql);
        $stmt->execute([
            'user_id' => $_SESSION['id'] ?? 0,
            'user_name' => $_SESSION['name'] ?? 'System',
            'action' => $action,
            'target_type' => $target_type,
            'target_id' => $target_id
        ]);
    } catch (PDOException $e) {
        // Fail silently or log to a file
        error_log("Audit Log Error: " . $e->getMessage());
    }
}

// ----------- APP DATA HELPERS -----------

/**
 * Returns all approved projects (id, name).
 */
function get_approved_projects(PDO $pdo): array {
    try {
        $stmt = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC");
        return $stmt->fetchAll();
    } catch (PDOException $e) {
        return [];
    }
}

/**
 * Returns a single news article by ID, or null if not found.
 */
function get_article_by_id(PDO $pdo, int $id): ?array {
    try {
        $stmt = $pdo->prepare("SELECT * FROM news_articles WHERE id = :id");
        $stmt->execute(['id' => $id]);
        $article = $stmt->fetch(PDO::FETCH_ASSOC);
        return $article ?: null;
    } catch (PDOException $e) {
        return null;
    }
}
/**
 * Fetches all key statistics for the dashboard KPIs in a single query.
 */
function get_dashboard_stats(PDO $pdo): array
{
    $stats = ['total' => 0, 'ongoing' => 0, 'completed' => 0, 'planned' => 0, 'stalled' => 0, 'pending_users' => 0, 'pending_projects' => 0];
    try {
        $stats_sql = "
            SELECT status as metric, COUNT(*) as value FROM projects WHERE is_approved = 1 GROUP BY status
            UNION ALL
            SELECT 'total' as metric, COUNT(*) as value FROM projects WHERE is_approved = 1
            UNION ALL
            SELECT 'pending_projects' as metric, COUNT(*) as value FROM projects WHERE is_approved = 0";

        if (is_super_admin()) {
            $stats_sql .= " UNION ALL SELECT 'pending_users' as metric, COUNT(*) as value FROM users WHERE is_active = 0";
        }
        $stmt = $pdo->query($stats_sql);
        $all_counts = $stmt->fetchAll(PDO::FETCH_KEY_PAIR);

        $stats['ongoing']   = (int)($all_counts['Ongoing'] ?? 0);
        $stats['completed'] = (int)($all_counts['Completed'] ?? 0);
        $stats['planned']   = (int)($all_counts['Planned'] ?? 0);
        $stats['stalled']   = (int)($all_counts['Stalled'] ?? 0);
        $stats['total']     = (int)($all_counts['total'] ?? 0);
        $stats['pending_projects'] = (int)($all_counts['pending_projects'] ?? 0);
        $stats['pending_users']    = (int)($all_counts['pending_users'] ?? 0);

    } catch (PDOException $e) {
        error_log("Dashboard Stats Error: " . $e->getMessage());
    }
    return $stats;
}

/**
 * Fetches a list of projects pending approval.
 */
function get_pending_approval_projects(PDO $pdo): array
{
    try {
        $sql = "SELECT p.id, p.name as project_name, w.name as ward_name, u.name as created_by_name
                FROM projects p
                LEFT JOIN wards w ON p.ward_id = w.id
                LEFT JOIN users u ON p.created_by = u.id
                WHERE p.is_approved = 0
                ORDER BY p.created_at DESC";
        return $pdo->query($sql)->fetchAll();
    } catch (PDOException $e) {
        error_log("Pending Projects Fetch Error: " . $e->getMessage());
        return [];
    }
}

/**
 * Fetches a paginated and filterable list of approved projects.
 * Uses LEFT JOIN so even if ward/constituency is missing, project is shown.
 */
function get_approved_projects_list(PDO $pdo, array $filters = []): array
{
    $page = $filters['page'] ?? 1;
    $per_page = $filters['per_page'] ?? 15;
    $offset = ($page - 1) * $per_page;

    $base_sql = "FROM projects p
                 LEFT JOIN wards w ON p.ward_id = w.id
                 LEFT JOIN constituencies c ON w.constituency_id = c.id
                 WHERE p.is_approved = 1";
    $params = [];

    if (!empty($filters['user_id'])) {
        $base_sql .= " AND p.created_by = :user_id";
        $params['user_id'] = $filters['user_id'];
    }
    if (!empty($filters['search'])) {
        $base_sql .= " AND p.name LIKE :search";
        $params['search'] = '%' . $filters['search'] . '%';
    }
    if (!empty($filters['status'])) {
        $base_sql .= " AND p.status = :status";
        $params['status'] = $filters['status'];
    }

    try {
        $count_stmt = $pdo->prepare("SELECT COUNT(p.id) " . $base_sql);
        $count_stmt->execute($params);
        $total_projects = $count_stmt->fetchColumn();

        $projects_sql = "SELECT
                            p.id,
                            p.name AS project_name,
                            p.status,
                            p.start_date,
                            p.end_date,
                            p.budget,
                            w.name AS ward_name,
                            c.name AS constituency_name
                       " . $base_sql
                       . " ORDER BY p.updated_at DESC LIMIT :limit OFFSET :offset";

        $projects_stmt = $pdo->prepare($projects_sql);

        foreach ($params as $key => $val) {
            $projects_stmt->bindValue(":$key", $val);
        }
        $projects_stmt->bindValue(':limit', (int)$per_page, PDO::PARAM_INT);
        $projects_stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);

        $projects_stmt->execute();

        return [
            'projects' => $projects_stmt->fetchAll(),
            'total_pages' => ceil($total_projects / $per_page),
            'total_projects' => (int)$total_projects,
            'offset' => (int)$offset,
            'per_page' => (int)$per_page
        ];

    } catch (PDOException $e) {
        error_log("Approved Projects Fetch Error: " . $e->getMessage());
        return ['projects' => [], 'total_pages' => 0, 'total_projects' => 0, 'offset' => 0, 'per_page' => $per_page];
    }
}

/**
 * Fetches the count of approved projects grouped by constituency.
 */
function get_projects_by_constituency(PDO $pdo): array
{
    $sql = "SELECT c.name, COUNT(p.id) as project_count
            FROM projects p
            LEFT JOIN wards w ON p.ward_id = w.id
            LEFT JOIN constituencies c ON w.constituency_id = c.id
            WHERE p.is_approved = 1
            GROUP BY c.id, c.name
            ORDER BY project_count DESC";
    try {
        return $pdo->query($sql)->fetchAll(PDO::FETCH_KEY_PAIR);
    } catch (PDOException $e) {
        error_log("Constituency Projects Fetch Error: " . $e->getMessage());
        return [];
    }
}

/**
 * Generates advanced pagination HTML for Projects.
 *
 * @param int $current_page The current page number.
 * @param int $total_pages The total number of pages.
 * @param array $params The current GET parameters for filters.
 * @param int $links_to_show The number of page links to show around the current page.
 * @return string The generated HTML for the pagination controls.
 */
function generate_pagination($current_page, $total_pages, $params = [], $links_to_show = 2) {
    if ($total_pages <= 1) {
        return '';
    }

    $base_class = "inline-flex items-center justify-center h-10 w-10 border text-sm font-medium transition-colors";
    $default_class = "bg-white dark:bg-dark-card border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700";
    $active_class = "bg-brand-orange border-brand-orange text-white z-10";
    $disabled_class = "bg-gray-100 dark:bg-gray-800 border-gray-300 dark:border-gray-700 text-gray-400 cursor-not-allowed";
    
    $html = '<nav class="flex items-center justify-between" aria-label="Pagination">';
    $html .= '<div class="hidden sm:block"><p class="text-sm text-gray-700 dark:text-gray-400">Page <span class="font-medium">' . $current_page . '</span> of <span class="font-medium">' . $total_pages . '</span></p></div>';
    $html .= '<div class="flex-1 flex justify-between sm:justify-end"><ul class="inline-flex items-center -space-x-px rounded-md shadow-sm">';

    // Previous Button
    if ($current_page > 1) {
        $prev_params = http_build_query(array_merge($params, ['page' => $current_page - 1]));
        $html .= '<li><a href="?' . $prev_params . '" class="' . $base_class . ' rounded-l-md">' . 'Previous</a></li>';
    } else {
        $html .= '<li><span class="' . $base_class . ' rounded-l-md ' . $disabled_class . '">Previous</span></li>';
    }

    // Page number links
    for ($i = 1; $i <= $total_pages; $i++) {
        if ($i == 1 || $i == $total_pages || ($i >= $current_page - $links_to_show && $i <= $current_page + $links_to_show)) {
            $page_params = http_build_query(array_merge($params, ['page' => $i]));
            $is_current = ($i == $current_page);
            $html .= '<li><a href="?' . $page_params . '" class="' . $base_class . ' ' . ($is_current ? $active_class : $default_class) . '" ' . ($is_current ? 'aria-current="page"' : '') . '>' . $i . '</a></li>';
        } elseif ($i == $current_page - ($links_to_show + 1) || $i == $current_page + ($links_to_show + 1)) {
            $html .= '<li><span class="' . $base_class . ' ' . $default_class . '">...</span></li>';
        }
    }

    // Next Button
    if ($current_page < $total_pages) {
        $next_params = http_build_query(array_merge($params, ['page' => $current_page + 1]));
        $html .= '<li><a href="?' . $next_params . '" class="' . $base_class . ' rounded-r-md">Next</a></li>';
    } else {
        $html .= '<li><span class="' . $base_class . ' rounded-r-md ' . $disabled_class . '">Next</span></li>';
    }

    $html .= '</ul></div></nav>';
    return $html;
}

