<?php
// FILE: admin/forgot_password.php
// PURPOSE: Allows users to request a password reset link.

require_once 'functions.php';
require_once 'db.php';

$errors = [];
$success_message = '';

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $email = trim($_POST['email']);

    if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors['email'] = "Please enter a valid email address.";
    } else {
        try {
            // Check if user exists
            $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :email");
            $stmt->execute(['email' => $email]);
            if ($stmt->fetch()) {
                // User exists, generate a token
                $token = bin2hex(random_bytes(50)); // Generate a secure random token
                $expires = new DateTime('NOW');
                $expires->add(new DateInterval('PT1H')); // Token expires in 1 hour

                // Delete any old tokens for this email
                $pdo->prepare("DELETE FROM password_resets WHERE email = :email")->execute(['email' => $email]);

                // Insert new token into the database
                $sql = "INSERT INTO password_resets (email, token, expires_at) VALUES (:email, :token, :expires_at)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'email' => $email,
                    'token' => $token,
                    'expires_at' => $expires->format('Y-m-d H:i:s')
                ]);

                // --- Send the email ---
                // IMPORTANT: Your server must be configured to send emails for this to work.
                $reset_link = "http://" . $_SERVER['HTTP_HOST'] . dirname($_SERVER['PHP_SELF']) . "/reset_password.php?token=" . $token;
                
                $subject = "Password Reset Request - Genowa Delivery Unit";
                $message = "
                    <html><body>
                    <p>Hello,</p>
                    <p>We received a request to reset your password for the Genowa Delivery Unit admin panel.</p>
                    <p>If you did not make this request, you can ignore this email.</p>
                    <p>To reset your password, please click the link below. This link is valid for one hour:</p>
                    <p><a href='{$reset_link}'>{$reset_link}</a></p>
                    <p>Thank you,</p>
                    <p>The Genowa Delivery Unit Team</p>
                    </body></html>
                ";
                $headers = "MIME-Version: 1.0" . "\r\n";
                $headers .= "Content-type:text/html;charset=UTF-8" . "\r\n";
                $headers .= 'From: <noreply@genowa.go.ke>' . "\r\n";

                if (mail($email, $subject, $message, $headers)) {
                    $success_message = "If an account with that email exists, a password reset link has been sent to it.";
                } else {
                    $errors['mail'] = "Could not send the password reset email. Please contact an administrator.";
                }

            } else {
                // To prevent user enumeration, show the same success message even if the email is not found.
                $success_message = "If an account with that email exists, a password reset link has been sent to it.";
            }
        } catch (PDOException $e) {
            $errors['db'] = "A database error occurred. Please try again later.";
        }
    }
}

include 'includes/header.php';
?>
<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        <div>
            <div class="text-center">
                 <h1 class="text-3xl font-extrabold text-brand-blue">GENOWA</h1>
                 <p class="text-lg font-semibold text-brand-orange tracking-wider">DELIVERY UNIT</p>
            </div>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Reset Your Password
            </h2>
            <p class="mt-2 text-center text-sm text-gray-600">
                Enter your email address and we will send you a link to reset your password.
            </p>
        </div>
        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert">
                    <p><?php echo e($success_message); ?></p>
                </div>
            <?php endif; ?>

            <?php if (!empty($errors)): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                    <?php foreach ($errors as $error): ?>
                        <p><?php echo e($error); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <form class="space-y-6" action="forgot_password.php" method="post">
                <div>
                    <label for="email" class="sr-only">Email address</label>
                    <input id="email" name="email" type="email" autocomplete="email" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Enter your email address">
                </div>
                 <div>
                    <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out">
                        Send Reset Link
                    </button>
                </div>
            </form>
             <p class="text-center text-sm text-gray-600">
                Remember your password? 
                <a href="login.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign in here
                </a>
            </p>
        </div>
    </div>
</div>
<?php include 'includes/footer.php'; ?>
