// FILE: /admin/assets/js/footer.js (NEW & COMPLETE - DEBUGGING & RESILIENT VERSION)
// PURPOSE: Controls the chat widget toggle with added error checking and self-contained animations.

document.addEventListener('DOMContentLoaded', () => {
    
    // --- CHAT WIDGET LOGIC ---
    const chatToggleButton = document.getElementById('chat-toggle-btn');
    const chatIframeContainer = document.getElementById('chat-iframe-container');
    const chatOpenIcon = document.getElementById('chat-open-icon');
    const chatCloseIcon = document.getElementById('chat-close-icon');
    let isChatIframeLoaded = false;

    // --- ENHANCEMENT: Check if all required elements exist before proceeding ---
    if (!chatToggleButton || !chatIframeContainer || !chatOpenIcon || !chatCloseIcon) {
        console.error('Genowa Chat Widget Error: One or more required HTML elements are missing. Please check the IDs in your footer.php file.');
        return; 
    }

    console.log('Genowa Chat Widget: All elements found and script initialized.');

    chatToggleButton.addEventListener('click', () => {
        try {
            console.log('Chat toggle button clicked.');
            const isPressed = chatToggleButton.getAttribute('aria-pressed') === 'true';
            console.log(`Current state (isPressed): ${isPressed}. Toggling to: ${!isPressed}`);
            
            // Toggle the visibility state
            chatToggleButton.setAttribute('aria-pressed', String(!isPressed));
            chatIframeContainer.classList.toggle('hidden');
            chatOpenIcon.classList.toggle('hidden');
            chatCloseIcon.classList.toggle('hidden');

            console.log(`Chat container 'hidden' class is now: ${chatIframeContainer.classList.contains('hidden')}`);

            // --- OPTIMIZATION: Lazy-load the iframe content ---
            if (!isPressed && !isChatIframeLoaded) {
                console.log('Lazy-loading iframe for the first time.');
                
                const loadingMessage = document.createElement('p');
                loadingMessage.id = 'chat-loading-message';
                loadingMessage.className = 'flex items-center justify-center h-full text-gray-500 p-4';
                loadingMessage.textContent = 'Loading Chat...';
                chatIframeContainer.appendChild(loadingMessage);

                const iframe = document.createElement('iframe');
                iframe.id = 'chat-iframe';
                iframe.src = 'chatbot.php';
                iframe.className = 'h-full w-full rounded-lg shadow-2xl border border-gray-300';
                iframe.title = 'Genowa Delivery Unit Chatbot';
                
                iframe.style.visibility = 'hidden';

                iframe.onload = () => {
                    console.log('Iframe has finished loading.');
                    const loader = document.getElementById('chat-loading-message');
                    if (loader) {
                        loader.remove();
                    }
                    iframe.style.visibility = 'visible';
                };
                
                chatIframeContainer.appendChild(iframe);
                isChatIframeLoaded = true;
            }
        } catch (error) {
            console.error('An error occurred in the chat toggle click handler:', error);
        }
    });

    // --- BACK TO TOP BUTTON LOGIC ---
    const backToTopButton = document.getElementById('back-to-top');
    if (backToTopButton) {
        window.addEventListener('scroll', () => {
            backToTopButton.classList.toggle('hidden', window.scrollY <= 300);
        });
        backToTopButton.addEventListener('click', () => {
            window.scrollTo({ top: 0, behavior: 'smooth' });
        });
    }

    // --- FOOTER FADE-IN ANIMATION (RESILIENT VERSION) ---
    // This version uses the Web Animations API, which is built into the browser
    // and does not depend on any external CSS files like the missing 'custom.css'.
    const footerSections = document.querySelectorAll('.footer-fade-in-section');
    if (footerSections.length > 0) {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    // Animate the element directly with JavaScript
                    entry.target.animate([
                        // Keyframes
                        { opacity: 0, transform: 'translateY(20px)' },
                        { opacity: 1, transform: 'translateY(0)' }
                    ], {
                        // Timing options
                        duration: 800,
                        easing: 'ease-out',
                        fill: 'forwards' // Keep the final state of the animation
                    });
                    
                    // Stop observing the element once it has been animated
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.1 }); // Trigger when 10% of the element is visible

        footerSections.forEach(section => {
            observer.observe(section);
        });
    }
});
