<?php
// FILE: admin/signup.php
// PURPOSE: Secure enterprise-grade user registration with enhanced validation, logging, and UI.

require_once 'config.php';
require_once 'db.php';
require_once 'includes/mailer_template.php';

// --- SESSION SECURITY ---
ini_set('session.cookie_httponly', 1);
if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
    ini_set('session.cookie_secure', 1);
}
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}
session_regenerate_id(true);

// --- CONSTANTS ---
define('MAX_REGISTRATION_ATTEMPTS', 5);
define('REGISTRATION_ATTEMPT_WINDOW', 60); // minutes

function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

// ... (Email notification helper remains unchanged) ...
function sendPendingApprovalEmail($recipient_email, $recipient_name, $config) {
    // ... (same as before) ...
    $subject = 'Registration Received - Awaiting Approval';
    $protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off' || $_SERVER['SERVER_PORT'] == 443) ? "https://" : "http://";
    $domainName = $_SERVER['HTTP_HOST'];
    $logoUrl = $protocol . $domainName . '/admin/assets/hbc_logo.png';
    $template_path = __DIR__ . '/email_template.html';
    if (!file_exists($template_path)) {
        error_log("Mailer Error: email_template.html not found at path: " . $template_path);
        return false;
    }
    $email_body = file_get_contents($template_path);
    $email_body = str_replace('{{recipient_name}}', e($recipient_name), $email_body);
    $email_body = str_replace('{{logo_url}}', $logoUrl, $email_body);
    $alt_body = "Dear " . e($recipient_name) . ",\n\nThank you for registering. Your account is now pending approval from an administrator. You will receive another email once your account is activated. You can then log in at: https://genowa.co.ke/admin/login.php\n\nRegards,\nGenowa Delivery Unit";
    return send_styled_email($recipient_email, $recipient_name, $subject, $email_body, $alt_body, $config);
}

$errors = [];
$success_message = '';
$inputs = [];
$is_ip_blocked = false;

try {
    $ip_address = $_SERVER['REMOTE_ADDR'];
    // Log every attempt (even failed) to improve brute force detection
    if ($_SERVER["REQUEST_METHOD"] == "POST") {
        $reg_stmt = $pdo->prepare("INSERT INTO registration_attempts (ip_address) VALUES (:ip_address)");
        $reg_stmt->execute(['ip_address' => $ip_address]);
    }
    $stmt = $pdo->prepare(
        "SELECT COUNT(*) FROM registration_attempts WHERE ip_address = :ip_address AND attempt_time > (NOW() - INTERVAL :window MINUTE)"
    );
    $stmt->execute(['ip_address' => $ip_address, 'window' => REGISTRATION_ATTEMPT_WINDOW]);
    $attempts_count = $stmt->fetchColumn();
    if ($attempts_count >= MAX_REGISTRATION_ATTEMPTS) {
        $is_ip_blocked = true;
        $errors['db'] = "You have made too many registration attempts. Please try again later.";
    }
} catch (PDOException $e) {
    error_log("Registration IP Check Failed: " . $e->getMessage());
    $errors['db'] = "A security check failed. Please try again later.";
    $is_ip_blocked = true;
}

try {
    $wards = $pdo->query("SELECT id, name FROM wards ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db'] = "Could not fetch ward list. Please contact an administrator.";
    error_log("Ward Fetch Failed: " . $e->getMessage());
}

if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// --- FORM SUBMISSION ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && !$is_ip_blocked) {
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $errors['db'] = "Invalid form submission. Please try again.";
        error_log("CSRF Token Mismatch from IP: $ip_address");
    } else {
        // Sanitize, normalize, and store inputs
        $inputs['name'] = trim($_POST['name']);
        $inputs['email'] = strtolower(trim($_POST['email']));
        $inputs['phone_number'] = preg_replace('/\s+/', '', trim($_POST['phone_number']));
        $inputs['ward_id'] = trim($_POST['ward_id']);
        $password = trim($_POST['password']);
        $password_confirm = trim($_POST['password_confirm']);

        // Phone number normalization
        if (preg_match('/^07\d{8}$/', $inputs['phone_number'])) {
            $inputs['phone_number'] = '+254' . substr($inputs['phone_number'], 1);
        }
        // Phone number validation (accepts +2547XXXXXXXX)
        if (!preg_match('/^\+2547\d{8}$/', $inputs['phone_number'])) {
            $errors['phone_number'] = "Enter a valid Kenyan mobile number (e.g. +254712345678 or 0712345678).";
        }

        // Basic validation
        if (empty($inputs['name'])) $errors['name'] = "Full name is required.";
        if (empty($inputs['email'])) {
            $errors['email'] = "Email is required.";
        } elseif (!filter_var($inputs['email'], FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = "Please enter a valid email address.";
        }
        if (empty($inputs['phone_number'])) {
            $errors['phone_number'] = "Phone number is required.";
        }
        if (empty($inputs['ward_id'])) $errors['ward_id'] = "Please select your assigned ward.";

        // Password complexity
        if (empty($password)) {
            $errors['password'] = "Password is required.";
        } elseif (strlen($password) < 8) {
            $errors['password'] = "Password must be at least 8 characters long.";
        } elseif (!preg_match('/[A-Z]/', $password)) {
            $errors['password'] = "Password must contain at least one uppercase letter.";
        } elseif (!preg_match('/[a-z]/', $password)) {
            $errors['password'] = "Password must contain at least one lowercase letter.";
        } elseif (!preg_match('/[0-9]/', $password)) {
            $errors['password'] = "Password must contain at least one number.";
        } elseif (!preg_match('/[\'^£$%&*()}{@#~?><>,|=_+¬-]/', $password)) {
            $errors['password'] = "Password must contain at least one special character.";
        }

        if ($password !== $password_confirm) $errors['password_confirm'] = "Passwords do not match.";

        // Database uniqueness check
        if (empty($errors)) {
            try {
                $stmt = $pdo->prepare("SELECT id FROM users WHERE email = :email");
                $stmt->execute(['email' => $inputs['email']]);
                if ($stmt->fetch()) {
                    $errors['email'] = "This email address is already registered.";
                }
                $stmt = $pdo->prepare("SELECT id FROM users WHERE phone_number = :phone_number");
                $stmt->execute(['phone_number' => $inputs['phone_number']]);
                if ($stmt->fetch()) {
                    $errors['phone_number'] = "This phone number is already registered.";
                }
            } catch (PDOException $e) {
                $errors['db'] = "A database error occurred during validation.";
                error_log("Signup Uniqueness Check PDOException: " . $e->getMessage());
            }
        }

        if (empty($errors)) {
            try {
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                $sql = "INSERT INTO users (name, email, phone_number, password, role, ward_id, is_active) VALUES (:name, :email, :phone_number, :password, 'Inputer', :ward_id, 0)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'name' => $inputs['name'],
                    'email' => $inputs['email'],
                    'phone_number' => $inputs['phone_number'],
                    'password' => $hashed_password,
                    'ward_id' => $inputs['ward_id']
                ]);
                if (sendPendingApprovalEmail($inputs['email'], $inputs['name'], $config)) {
                    // Success: redirect to login after a short pause
                    $_SESSION['signup_success'] = "Registration successful! Your account is now pending approval. A confirmation email has been sent to you.";
                } else {
                    $_SESSION['signup_success'] = "Registration successful! Your account is now pending approval. We could not send a confirmation email at this time.";
                    error_log("Successful registration but FAILED TO SEND EMAIL for user '{$inputs['email']}' from IP: $ip_address");
                }
                error_log("Successful Registration for user '{$inputs['email']}' from IP: $ip_address");
                $inputs = [];
                unset($_SESSION['csrf_token']);
                header('Location: login.php?signup=1'); // Redirect after POST (login.php will display session message)
                exit;
            } catch (PDOException $e) {
                $errors['db'] = "An error occurred during registration. Please try again.";
                error_log("Signup Insert PDOException: " . $e->getMessage());
            }
        } else {
            error_log("Failed Registration attempt for user '{$inputs['email']}' from IP: $ip_address. Errors: " . implode(', ', $errors));
        }
    }
}

include 'includes/header.php';
?>

<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        <div>
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Create New Account
            </h2>
            <p class="mt-2 text-center text-sm text-gray-600">
                Sign up to access the admin section of the Genowa Delivery Portal
            </p>
        </div>

        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md flex items-center" role="alert">
                    <svg class="h-5 w-5 text-green-600 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><title>Success</title><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"/></svg>
                    <div>
                        <p class="font-bold">Success</p>
                        <p><?php echo e($success_message); ?></p>
                    </div>
                </div>
            <?php endif; ?>

            <?php if (isset($errors['db'])): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md flex items-center" role="alert">
                    <svg class="h-5 w-5 text-red-600 mr-2" fill="none" stroke="currentColor" viewBox="0 0 24 24"><title>Error</title><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/></svg>
                    <div>
                        <p><?php echo e($errors['db']); ?></p>
                    </div>
                </div>
            <?php endif; ?>

            <form id="signup-form" class="space-y-6" action="signup.php" method="post" novalidate autocomplete="off">
                <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">

                <div>
                    <label for="name" class="block text-sm font-medium text-gray-700">Full Name</label>
                    <input id="name" name="name" type="text" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['name']) ? 'border-red-500' : 'border-gray-300'; ?> placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Jane Akinyi" value="<?php echo e($inputs['name'] ?? ''); ?>" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                    <?php if(isset($errors['name'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['name']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="email" class="block text-sm font-medium text-gray-700">Email Address</label>
                    <input id="email" name="email" type="email" autocomplete="email" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['email']) ? 'border-red-500' : 'border-gray-300'; ?> placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="you@example.com" value="<?php echo e($inputs['email'] ?? ''); ?>" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                    <?php if(isset($errors['email'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['email']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="phone_number" class="block text-sm font-medium text-gray-700">Phone Number</label>
                    <input id="phone_number" name="phone_number" type="tel" autocomplete="tel" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['phone_number']) ? 'border-red-500' : 'border-gray-300'; ?> placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="e.g. +254712345678" value="<?php echo e($inputs['phone_number'] ?? ''); ?>" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                    <?php if(isset($errors['phone_number'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['phone_number']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="ward_id" class="block text-sm font-medium text-gray-700">Assigned Ward</label>
                    <select name="ward_id" id="ward_id" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['ward_id']) ? 'border-red-500' : 'border-gray-300'; ?> bg-white text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                        <option value="">-- Select Your Ward --</option>
                        <?php if (!empty($wards)): ?>
                            <?php foreach ($wards as $ward): ?>
                                <option value="<?php echo e($ward['id']); ?>" <?php echo (($inputs['ward_id'] ?? '') == $ward['id']) ? 'selected' : ''; ?>>
                                    <?php echo e($ward['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        <?php endif; ?>
                    </select>
                    <?php if(isset($errors['ward_id'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['ward_id']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="password" class="block text-sm font-medium text-gray-700">Password <span class="text-xs text-gray-500">(8+ chars, uppercase, lowercase, number &amp; symbol)</span></label>
                    <div class="relative">
                        <input id="password" name="password" type="password" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['password']) ? 'border-red-500' : 'border-gray-300'; ?> placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm pr-10" placeholder="8+ characters, with uppercase, number, & symbol" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                        <button type="button" tabindex="-1" onclick="togglePassword('password')" class="absolute right-2 top-1/2 -translate-y-1/2 text-gray-500 hover:text-brand-purple focus:outline-none" aria-label="Show/Hide password"><svg id="eye-password" class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 01-6 0 3 3 0 016 0zm-9 0c0-5.25 8.25-9 9-9s9 3.75 9 9-8.25 9-9 9-9-3.75-9-9z"/></svg></button>
                    </div>
                    <div id="password-strength" class="mt-1 text-xs text-gray-500"></div>
                    <?php if(isset($errors['password'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password']; ?></p><?php endif; ?>
                </div>

                <div>
                    <label for="password_confirm" class="block text-sm font-medium text-gray-700">Confirm Password</label>
                    <div class="relative">
                        <input id="password_confirm" name="password_confirm" type="password" required class="mt-1 appearance-none relative block w-full px-3 py-3 border <?php echo isset($errors['password_confirm']) ? 'border-red-500' : 'border-gray-300'; ?> placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm pr-10" placeholder="Re-enter your password" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-required="true">
                        <button type="button" tabindex="-1" onclick="togglePassword('password_confirm')" class="absolute right-2 top-1/2 -translate-y-1/2 text-gray-500 hover:text-brand-purple focus:outline-none" aria-label="Show/Hide password"><svg id="eye-password_confirm" class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 12a3 3 0 01-6 0 3 3 0 016 0zm-9 0c0-5.25 8.25-9 9-9s9 3.75 9 9-8.25 9-9 9-9-3.75-9-9z"/></svg></button>
                    </div>
                    <?php if(isset($errors['password_confirm'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password_confirm']; ?></p><?php endif; ?>
                </div>

                <div>
                    <button id="register-button" type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out disabled:bg-orange-300 disabled:cursor-not-allowed" <?php if($is_ip_blocked) echo 'disabled'; ?> aria-disabled="<?php echo $is_ip_blocked ? 'true' : 'false'; ?>">
                        <span class="button-text">Register</span>
                        <div class="loader ease-linear rounded-full border-4 border-t-4 border-gray-200 h-5 w-5 ml-3 hidden" aria-live="polite"></div>
                    </button>
                </div>
            </form>
            <p class="text-center text-sm text-gray-600">
                Already have an account?
                <a href="login.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign in here
                </a>
            </p>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const signupForm = document.getElementById('signup-form');
    const registerButton = document.getElementById('register-button');
    const buttonText = registerButton.querySelector('.button-text');
    const loader = registerButton.querySelector('.loader');

    if (signupForm) {
        signupForm.addEventListener('submit', function(e) {
            registerButton.disabled = true;
            buttonText.classList.add('hidden');
            loader.classList.remove('hidden');
        });
    }

    // Password show/hide toggle
    window.togglePassword = function(field) {
        const input = document.getElementById(field);
        if (!input) return;
        if (input.type === "password") {
            input.type = "text";
        } else {
            input.type = "password";
        }
    };

    // Password strength meter
    const pwdInput = document.getElementById('password');
    const strengthBar = document.getElementById('password-strength');
    if (pwdInput && strengthBar) {
        pwdInput.addEventListener('input', function() {
            const val = pwdInput.value;
            let score = 0;
            if (val.length >= 8) score++;
            if (/[A-Z]/.test(val)) score++;
            if (/[a-z]/.test(val)) score++;
            if (/[0-9]/.test(val)) score++;
            if (/[\'^£$%&*()}{@#~?><>,|=_+¬-]/.test(val)) score++;
            let strengthText = '';
            switch (score) {
                case 0:
                case 1:
                    strengthText = 'Too short';
                    strengthBar.style.color = '#dc2626';
                    break;
                case 2:
                    strengthText = 'Weak';
                    strengthBar.style.color = '#f59e42';
                    break;
                case 3:
                    strengthText = 'Moderate';
                    strengthBar.style.color = '#eab308';
                    break;
                case 4:
                    strengthText = 'Strong';
                    strengthBar.style.color = '#22c55e';
                    break;
                case 5:
                    strengthText = 'Very strong';
                    strengthBar.style.color = '#22d3ee';
                    break;
            }
            strengthBar.textContent = strengthText;
        });
    }
});
</script>

<style>
.loader {
    border-top-color: #f97316; /* brand-orange */
    -webkit-animation: spinner 1.5s linear infinite;
    animation: spinner 1.5s linear infinite;
}
@-webkit-keyframes spinner {
    0% { -webkit-transform: rotate(0deg); }
    100% { -webkit-transform: rotate(360deg); }
}
@keyframes spinner {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}
</style>

<?php
include 'includes/footer.php';
?>
