<?php
// FILE: admin/reset_password.php
// PURPOSE: Handles the final step of the password reset process after verifying the token.

require_once 'functions.php';
require_once 'db.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

$errors = [];
$success_message = '';
$token = $_GET['token'] ?? '';
$token_is_valid = false;
$user_email = '';

// --- GENERATE CSRF TOKEN ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

if (empty($token)) {
    $errors['token'] = "No reset token provided. Please request a new link.";
} else {
    try {
        // Hash the token from the URL to match what's in the database
        $token_hash = hash('sha256', $token);

        // Check if the hashed token is valid and not expired
        $stmt = $pdo->prepare("SELECT email FROM password_resets WHERE token = :token AND expires_at > NOW()");
        $stmt->execute(['token' => $token_hash]);
        $reset_request = $stmt->fetch();

        if ($reset_request) {
            $token_is_valid = true;
            $user_email = $reset_request['email'];
        } else {
            $errors['token'] = "This password reset link is invalid or has expired. Please request a new one.";
        }

    } catch (PDOException $e) {
        $errors['db'] = "A database error occurred.";
        error_log("Reset Password Token Check PDOException: " . $e->getMessage());
    }
}

// Handle form submission for new password
if ($_SERVER["REQUEST_METHOD"] == "POST" && $token_is_valid) {
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $errors['db'] = "Invalid request. Please try again.";
    } else {
        $password = trim($_POST['password']);
        $password_confirm = trim($_POST['password_confirm']);

        // Password Complexity Validation
        if (empty($password)) {
            $errors['password'] = "Password is required.";
        } elseif (strlen($password) < 8) {
            $errors['password'] = "Password must be at least 8 characters long.";
        } elseif (!preg_match('/[A-Z]/', $password)) {
            $errors['password'] = "Password must contain at least one uppercase letter.";
        } elseif (!preg_match('/[a-z]/', $password)) {
            $errors['password'] = "Password must contain at least one lowercase letter.";
        } elseif (!preg_match('/[0-9]/', $password)) {
            $errors['password'] = "Password must contain at least one number.";
        } elseif (!preg_match('/[\'^£$%&*()}{@#~?><>,|=_+¬-]/', $password)) {
            $errors['password'] = "Password must contain at least one special character.";
        }

        if ($password !== $password_confirm) $errors['password_confirm'] = "Passwords do not match.";

        if (empty($errors)) {
            try {
                $pdo->beginTransaction();

                // Hash the new password
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);

                // Update the user's password in the users table
                $update_stmt = $pdo->prepare("UPDATE users SET password = :password WHERE email = :email");
                $update_stmt->execute(['password' => $hashed_password, 'email' => $user_email]);

                // Delete the used token from the password_resets table
                $delete_stmt = $pdo->prepare("DELETE FROM password_resets WHERE email = :email");
                $delete_stmt->execute(['email' => $user_email]);

                $pdo->commit();
                $success_message = "Your password has been reset successfully! You can now sign in with your new password.";
                $token_is_valid = false; // Prevent form from showing again

            } catch (PDOException $e) {
                $pdo->rollBack();
                $errors['db'] = "An error occurred while resetting your password. Please try again.";
                error_log("Reset Password Update PDOException: " . $e->getMessage());
            }
        }
    }
}

include 'includes/header.php';
?>
<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        <div>
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Set New Password
            </h2>
        </div>
        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert">
                    <p><?php echo e($success_message); ?></p>
                    <a href="login.php" class="font-bold text-green-800 hover:underline">Click here to sign in</a>
                </div>
            <?php elseif (!$token_is_valid): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                     <?php foreach ($errors as $error): ?><p><?php echo e($error); ?></p><?php endforeach; ?>
                </div>
            <?php else: ?>
                <form class="space-y-6" action="reset_password.php?token=<?php echo e($token); ?>" method="post">
                     <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                     <div>
                        <label for="password" class="sr-only">New Password</label>
                        <input id="password" name="password" type="password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Enter new password (min. 8 characters)">
                        <?php if(isset($errors['password'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password']; ?></p><?php endif; ?>
                    </div>
                    <div>
                        <label for="password_confirm" class="sr-only">Confirm New Password</label>
                        <input id="password_confirm" name="password_confirm" type="password" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Confirm new password">
                         <?php if(isset($errors['password_confirm'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['password_confirm']; ?></p><?php endif; ?>
                    </div>
                    <div>
                        <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple">
                            Reset Password
                        </button>
                    </div>
                </form>
            <?php endif; ?>
        </div>
    </div>
</div>
<?php include 'includes/footer.php'; ?>
