<?php
// FILE: admin/manage_project.php
// PURPOSE: Handles creating and editing projects with enterprise-grade UI and set-featured-photo support.
// VERSION: 6.2 - Improved UI, visible black borders, and set featured image button.

require_once 'functions.php';
require_once 'db.php';
require_login();

// --- Permissions Check ---
if (!is_inputer() && !is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- Mode Detection & Data Fetching ---
$project_id = $_GET['id'] ?? null;
$is_editing = $project_id !== null;
$page_title = $is_editing ? "Edit Project" : "Add New Project";
$project = [];
$project_photos = [];
$approval_comments = [];
$errors = [];

try {
    $pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY title")->fetchAll();
    $wards = $pdo->query("SELECT w.id, w.name, c.name as constituency_name FROM wards w JOIN constituencies c ON w.constituency_id = c.id ORDER BY c.name, w.name")->fetchAll();
} catch (PDOException $e) {
    die("Error fetching form data: " . $e->getMessage());
}

if ($is_editing) {
    $stmt = $pdo->prepare("SELECT * FROM projects WHERE id = :id");
    $stmt->execute(['id' => $project_id]);
    $project = $stmt->fetch();
    if (!$project) {
        $_SESSION['flash_message'] = "Error: Project not found.";
        $_SESSION['flash_type'] = 'error';
        header("location: admin_project_hub.php");
        exit;
    }
    if (!empty($project['latitude']) && !empty($project['longitude'])) {
        $project['coordinates'] = $project['latitude'] . ', ' . $project['longitude'];
    }

    $photos_stmt = $pdo->prepare("SELECT id, image_url, caption, is_featured, photo_type FROM project_photos WHERE project_id = :id ORDER BY is_featured DESC, id DESC");
    $photos_stmt->execute(['id' => $project_id]);
    $project_photos = $photos_stmt->fetchAll();
    $comments_stmt = $pdo->prepare("SELECT pac.*, u.name as approver_name FROM project_approval_comments pac JOIN users u ON pac.user_id = u.id WHERE pac.project_id = :id ORDER BY pac.created_at DESC");
    $comments_stmt->execute(['id' => $project_id]);
    $approval_comments = $comments_stmt->fetchAll();
}

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = filter_input(INPUT_POST, 'name', FILTER_SANITIZE_SPECIAL_CHARS);
    $description = filter_input(INPUT_POST, 'description', FILTER_SANITIZE_SPECIAL_CHARS);
    $beneficiaries = filter_input(INPUT_POST, 'beneficiaries_summary', FILTER_SANITIZE_SPECIAL_CHARS);
    $contractor_name = filter_input(INPUT_POST, 'contractor_name', FILTER_SANITIZE_SPECIAL_CHARS);

    $ward_id = filter_input(INPUT_POST, 'ward_id', FILTER_VALIDATE_INT);
    $pillar_id = filter_input(INPUT_POST, 'manifesto_pillar_id', FILTER_VALIDATE_INT);
    $status = in_array($_POST['status'], ['Planned', 'Ongoing', 'Completed', 'Stalled']) ? $_POST['status'] : 'Planned';
    $budget_allocated = filter_var($_POST['budget_allocated'], FILTER_VALIDATE_FLOAT, FILTER_NULL_ON_FAILURE) ?? 0.00;
    $budget_spent = filter_var($_POST['budget_spent'], FILTER_VALIDATE_FLOAT, FILTER_NULL_ON_FAILURE) ?? 0.00;
    $start_date = !empty($_POST['start_date']) ? date('Y-m-d', strtotime($_POST['start_date'])) : null;
    $expected_completion_date = !empty($_POST['expected_completion_date']) ? date('Y-m-d', strtotime($_POST['expected_completion_date'])) : null;
    $manual_progress = filter_input(INPUT_POST, 'manual_progress', FILTER_VALIDATE_INT, ["options" => ["min_range" => 0, "max_range" => 100]]);
    if ($manual_progress === false) $manual_progress = null;

    $coordinates_raw = trim($_POST['coordinates'] ?? '');
    $latitude = null;
    $longitude = null;
    if (!empty($coordinates_raw)) {
        $coords = explode(',', str_replace(['(', ')'], '', $coordinates_raw));
        if (count($coords) === 2) {
            $lat_candidate = filter_var(trim($coords[0]), FILTER_VALIDATE_FLOAT);
            $lon_candidate = filter_var(trim($coords[1]), FILTER_VALIDATE_FLOAT);
            if ($lat_candidate !== false && $lon_candidate !== false) {
                $latitude = $lat_candidate;
                $longitude = $lon_candidate;
            } else {
                $errors[] = "The provided GPS coordinates are in an invalid format. Please use 'latitude, longitude'.";
            }
        } else {
            $errors[] = "The provided GPS coordinates are in an invalid format. Please use 'latitude, longitude'.";
        }
    }

    if (empty($name)) $errors[] = "Project Name is required.";
    if (empty($description)) $errors[] = "Project Description is required.";
    if (empty($ward_id)) $errors[] = "Ward is required.";
    if (empty($pillar_id)) $errors[] = "Manifesto Pillar is required.";
    if ($budget_spent > $budget_allocated) $errors[] = "Budget Spent cannot be greater than Budget Allocated.";
    if ($start_date && $expected_completion_date && $expected_completion_date < $start_date) $errors[] = "Expected Completion Date cannot be before Start Date.";

    if (empty($errors)) {
        try {
            $pdo->beginTransaction();

            if ($is_editing) {
                $sql = "UPDATE projects SET 
                            name=:name, description=:desc, status=:status, manual_progress=:progress, 
                            budget_allocated=:alloc, budget_spent=:spent, start_date=:start, 
                            expected_completion_date=:end, beneficiaries_summary=:benef, 
                            contractor_name=:contractor, ward_id=:ward, manifesto_pillar_id=:pillar, 
                            latitude=:lat, longitude=:lon, last_updated_by=:user, 
                            is_approved=:is_approved 
                        WHERE id=:id";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([
                    'name' => $name, 'desc' => $description, 'status' => $status, 
                    'progress' => $manual_progress, 'alloc' => $budget_allocated, 'spent' => $budget_spent, 
                    'start' => $start_date, 'end' => $expected_completion_date, 'benef' => $beneficiaries, 
                    'contractor' => $contractor_name, 'ward' => $ward_id, 'pillar' => $pillar_id,
                    'lat' => $latitude, 'lon' => $longitude, 'user' => $_SESSION['id'], 
                    'is_approved' => ($project['is_approved'] == 2) ? 0 : $project['is_approved'], 
                    'id' => $project_id
                ]);
                log_action($pdo, 'Updated Project', 'Project', $project_id);
                $_SESSION['flash_message'] = "Project successfully updated! You can now add another project.";
                $_SESSION['flash_type'] = 'success';
            } else {
                $sql = "INSERT INTO projects (name, description, status, manual_progress, budget_allocated, budget_spent, start_date, expected_completion_date, beneficiaries_summary, contractor_name, ward_id, manifesto_pillar_id, latitude, longitude, created_by, last_updated_by) VALUES (?,?,?,?,?,?,?,?,?,?,?,?,?,?,?,?)";
                $stmt = $pdo->prepare($sql);
                $stmt->execute([$name, $description, $status, $manual_progress, $budget_allocated, $budget_spent, $start_date, $expected_completion_date, $beneficiaries, $contractor_name, $ward_id, $pillar_id, $latitude, $longitude, $_SESSION['id'], $_SESSION['id']]);
                $project_id = $pdo->lastInsertId();
                log_action($pdo, 'Created Project', 'Project', $project_id);
                $_SESSION['flash_message'] = "New project successfully created! You can now add another.";
                $_SESSION['flash_type'] = 'success';
            }

            if (isset($_FILES['project_photos']) && !empty(array_filter($_FILES['project_photos']['name']))) {
                $upload_dir = 'uploads/';
                $allowed_mime_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
                $photo_count = 0;

                foreach ($_FILES['project_photos']['name'] as $key => $filename) {
                    if (!empty($filename) && $_FILES['project_photos']['error'][$key] === UPLOAD_ERR_OK) {
                        $tmp_name = $_FILES['project_photos']['tmp_name'][$key];
                        $file_type = mime_content_type($tmp_name);
                        $photo_type_from_form = $_POST['photo_type'][$key] ?? 'Progress';
                        $photo_caption_from_form = filter_input(INPUT_POST, 'photo_caption', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY)[$key] ?? '';
                        $valid_photo_types = ['Before', 'Progress', 'After'];
                        
                        if (in_array($file_type, $allowed_mime_types) && in_array($photo_type_from_form, $valid_photo_types)) {
                            $unique_filename = "proj_{$project_id}_" . uniqid() . '.' . pathinfo($filename, PATHINFO_EXTENSION);
                            if (move_uploaded_file($tmp_name, $upload_dir . $unique_filename)) {
                                $photo_sql = "INSERT INTO project_photos (project_id, image_url, photo_type, caption) VALUES (?, ?, ?, ?)";
                                $pdo->prepare($photo_sql)->execute([$project_id, $unique_filename, $photo_type_from_form, $photo_caption_from_form]);
                                $photo_count++;
                            }
                        }
                    }
                }
                if ($photo_count > 0) {
                    log_action($pdo, 'Uploaded Photos', 'Project', $project_id, "Added {$photo_count} new photo(s).");
                }
            }
            
            $pdo->commit();
            header("Location: manage_project.php");
            exit;
        } catch (PDOException $e) {
            $pdo->rollBack();
            $_SESSION['flash_message'] = "Database Error: Could not save the project. " . $e->getMessage();
            $_SESSION['flash_type'] = 'error';
        }
    }
    $project = $_POST;
}

include 'includes/header.php';
?>
<style>
    .project-form-card { border: 2.5px solid #171717 !important; border-radius: 1.25rem !important; background: #fff !important; box-shadow: 0 6px 40px 0 rgba(20, 20, 20, 0.08); padding: 2.5rem; }
    .form-input, .form-textarea, .form-select { display: block; width: 100%; font-size: 1.1rem; line-height: 1.5; padding: 0.85rem 1rem; border: 2px solid #111 !important; border-radius: 0.7rem; background: #fff; color: #18181b; box-shadow: none; outline: none; transition: border-color .15s, box-shadow .15s; margin-top: 0.1rem; margin-bottom: 0.1rem;}
    .form-input:focus, .form-textarea:focus, .form-select:focus { border-color: #7c3aed !important; box-shadow: 0 0 0 2px rgba(124,58,237,.15); background: #fcfcfc; }
    .form-label { display: block; font-weight: 600; color: #18181b; font-size: 1rem; margin-bottom: .25rem; margin-top: .7rem; letter-spacing: .01em; }
    .form-label .text-red-500 { font-size: 1.05em; }
    .tab-button { padding: .5rem 1.5rem; font-size: 1.13rem; font-weight: 600; color: #636363; border-bottom: 3.5px solid transparent; background: none; outline: none; transition: all .22s cubic-bezier(.4,0,.2,1); border-radius: 0.7rem 0.7rem 0 0; margin-bottom: -1px; cursor: pointer; }
    .tab-button.active, .tab-button[aria-selected="true"] { color: #5128bb; border-bottom: 3.5px solid #111; background: #f6f6ff;}
    .tab-button:hover:not(.active) { background: #f3f3fa; color: #1e1e1e; }
    .form-section-divider { border-top: 2.5px dashed #171717; margin: 2.5rem 0 2.5rem 0; }
    .error-message, .flash-message { border-radius: .75rem; padding: 1rem 1.5rem; font-size: 1.07rem; margin-bottom: 2rem;}
    .error-message { background: #fef2f2; color: #b91c1c; border: 2px solid #fca5a5;}
    .flash-message.success { background: #e6f9ef; color: #15803d; border: 2px solid #4ade80;}
    .flash-message.error { background: #fef2f2; color: #b91c1c; border: 2px solid #fca5a5;}
</style>

<header class="bg-white shadow-sm">
  <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
    <h1 class="text-2xl font-bold leading-6 text-gray-900"><?php echo e($page_title); ?></h1>
    <p class="mt-1 text-base text-gray-500"><?php echo $is_editing ? 'Update the details for this project.' : 'Fill out the form to create a new project.'; ?></p>
  </div>
</header>

<main class="mx-auto max-w-3xl md:max-w-5xl px-4 sm:px-6 lg:px-8 py-8">
    <?php if (!empty($errors)): ?>
        <div class="error-message">
            <strong>Please correct the following errors:</strong>
            <ul class="list-disc list-inside mt-2">
                <?php foreach($errors as $error): ?>
                    <li><?php echo e($error); ?></li>
                <?php endforeach; ?>
            </ul>
        </div>
    <?php elseif (isset($_SESSION['flash_message'])): ?>
        <div class="flash-message <?php echo ($_SESSION['flash_type'] ?? 'success') === 'success' ? 'success' : 'error'; ?>">
            <?php echo e($_SESSION['flash_message']); unset($_SESSION['flash_message']); unset($_SESSION['flash_type']); ?>
        </div>
    <?php endif; ?>

    <?php if ($is_editing && $project['is_approved'] == 2 && !empty($approval_comments)): ?>
        <div class="mb-6 p-4 rounded-lg bg-yellow-100 border-l-4 border-yellow-500">
            <h3 class="text-lg font-bold text-yellow-800">Changes Requested</h3>
            <p class="text-base text-yellow-700 mt-1">By: <strong><?php echo e($approval_comments[0]['approver_name']); ?></strong> on <?php echo date('M j, Y', strtotime($approval_comments[0]['created_at'])); ?></p>
            <blockquote class="mt-2 pl-4 border-l-2 border-yellow-600 text-yellow-800" style="font-size:1.08rem">"<?php echo nl2br(e($approval_comments[0]['comment'])); ?>"</blockquote>
            <p class="mt-4 text-base font-semibold text-yellow-900">Please address the comments above and click "Save Changes" to re-submit for approval.</p>
        </div>
    <?php endif; ?>

    <form action="manage_project.php<?php echo $is_editing ? '?id=' . e($project_id) : ''; ?>" method="POST" enctype="multipart/form-data">
        <div x-data="photoTabs()" x-init="initPhotos()">
            <!-- Tab Navigation -->
            <div class="border-b border-gray-300 mb-8 flex flex-wrap gap-2 bg-white px-1 pt-2 rounded-t-xl">
                <button type="button" @click="activeTab = 1" :aria-selected="activeTab === 1" :class="{ 'active': activeTab === 1 }" class="tab-button">1. Core & Location</button>
                <button type="button" @click="activeTab = 2" :aria-selected="activeTab === 2" :class="{ 'active': activeTab === 2 }" class="tab-button">2. Photos</button>
                <button type="button" @click="activeTab = 3" :aria-selected="activeTab === 3" :class="{ 'active': activeTab === 3 }" class="tab-button">3. Status & Financials</button>
            </div>

            <!-- Tab Content -->
            <div class="project-form-card">
                <!-- Tab 1: Core Info & Location -->
                <div x-show="activeTab === 1" class="space-y-8" x-cloak>
                    <div>
                        <h2 class="text-xl font-bold text-gray-900 mb-2">Core Information</h2>
                        <div class="grid grid-cols-1 gap-y-8">
                            <div>
                                <label for="name" class="form-label">Project Name <span class="text-red-500">*</span></label>
                                <input type="text" id="name" name="name" required class="form-input" autocomplete="off" placeholder="Enter the official project name" value="<?php echo e($project['name'] ?? ''); ?>">
                            </div>
                            <div>
                                <label for="description" class="form-label">Project Description <span class="text-red-500">*</span></label>
                                <textarea id="description" name="description" rows="5" required class="form-textarea" placeholder="Describe the project's goals, scope, and impact..."><?php echo e($project['description'] ?? ''); ?></textarea>
                            </div>
                            <div>
                                <label for="beneficiaries_summary" class="form-label">Beneficiaries Summary</label>
                                <textarea id="beneficiaries_summary" name="beneficiaries_summary" rows="3" class="form-textarea" placeholder="Who will benefit from this project?"><?php echo e($project['beneficiaries_summary'] ?? ''); ?></textarea>
                            </div>
                        </div>
                    </div>
                    <div class="form-section-divider"></div>
                    <div>
                        <h2 class="text-xl font-bold text-gray-900 mb-2">Categorization & Mapping</h2>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                             <div>
                                 <label for="manifesto_pillar_id" class="form-label">Manifesto Pillar <span class="text-red-500">*</span></label>
                                 <select id="manifesto_pillar_id" name="manifesto_pillar_id" required class="form-select">
                                     <option value="">Select a Pillar</option>
                                     <?php foreach($pillars as $pillar): ?>
                                         <option value="<?php echo e($pillar['id']); ?>" <?php echo (isset($project['manifesto_pillar_id']) && $project['manifesto_pillar_id'] == $pillar['id']) ? 'selected' : ''; ?>><?php echo e($pillar['title']); ?></option>
                                     <?php endforeach; ?>
                                 </select>
                            </div>
                             <div>
                                <label for="ward_id" class="form-label">Ward <span class="text-red-500">*</span></label>
                                 <select id="ward_id" name="ward_id" required class="form-select">
                                     <option value="">Select a Ward</option>
                                     <?php $current_constituency = ''; ?>
                                     <?php foreach($wards as $ward): ?>
                                         <?php if ($ward['constituency_name'] !== $current_constituency): ?>
                                             <?php if ($current_constituency !== ''): ?></optgroup><?php endif; ?>
                                             <optgroup label="<?php echo e($ward['constituency_name']); ?>">
                                             <?php $current_constituency = $ward['constituency_name']; ?>
                                         <?php endif; ?>
                                         <option value="<?php echo e($ward['id']); ?>" <?php echo (isset($project['ward_id']) && $project['ward_id'] == $ward['id']) ? 'selected' : ''; ?>><?php echo e($ward['name']); ?></option>
                                     <?php endforeach; ?>
                                     </optgroup>
                                 </select>
                            </div>
                            <div class="md:col-span-2">
                                <label for="coordinates" class="form-label">GPS Coordinates</label>
                                <input type="text" id="coordinates" name="coordinates" class="form-input" autocomplete="off" placeholder="Paste coordinates from Google Maps, e.g., -0.4384, 34.5244" value="<?php echo e($project['coordinates'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tab 2: Photos -->
                <div x-show="activeTab === 2" x-data x-cloak>
                    <h2 class="text-xl font-bold text-gray-900 mb-2">Photo Management</h2>
                    <div class="mt-2">
                        <?php if($is_editing && !empty($project_photos)): ?>
                            <h3 class="text-base font-semibold text-gray-700 mb-2">Existing Photos</h3>
                            <div id="photo-gallery" class="grid grid-cols-2 sm:grid-cols-3 lg:grid-cols-4 gap-4 mb-6">
                                <?php foreach($project_photos as $photo): ?>
                                    <div class="relative group" id="photo-<?php echo e($photo['id']); ?>">
                                        <img src="uploads/<?php echo e($photo['image_url']); ?>" alt="<?php echo e($photo['caption'] ?? 'Project Photo'); ?>" class="rounded-md object-cover h-32 w-full border-2 border-black bg-gray-100">

                                        <!-- Featured Star Badge -->
                                        <?php if($photo['is_featured']): ?>
                                            <span class="absolute top-1 left-1 bg-yellow-300 text-yellow-900 font-bold px-2 py-1 text-xs rounded shadow-lg z-10">★ Featured</span>
                                        <?php endif; ?>

                                        <div class="absolute top-1 right-1 flex flex-col items-end gap-1 z-10">
                                            <!-- Favorite (Star) Button -->
                                            <button type="button"
                                                x-data
                                                :disabled="window.settingFeatured"
                                                title="Set as Featured"
                                                @click.prevent="window.setAsFeaturedPhoto(<?php echo e($photo['id']); ?>, <?php echo e($project_id); ?>)"
                                                class="bg-white border-2 border-yellow-400 text-yellow-500 rounded-full p-1 shadow hover:bg-yellow-100 transition 
                                                    <?php if($photo['is_featured']): ?> opacity-60 cursor-default <?php else: ?> opacity-90 hover:opacity-100 <?php endif; ?>">
                                                <?php if($photo['is_featured']): ?>
                                                    <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 20 20">
                                                        <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.286 3.978a1 1 0 00.95.69h4.21c.969 0 1.371 1.24.588 1.81l-3.404 2.475a1 1 0 00-.364 1.118l1.285 3.978c.3.921-.755 1.688-1.54 1.118l-3.404-2.475a1 1 0 00-1.175 0l-3.404 2.475c-.785.57-1.84-.197-1.54-1.118l1.285-3.978a1 1 0 00-.364-1.118L2.227 9.405c-.783-.57-.38-1.81.588-1.81h4.21a1 1 0 00.95-.69l1.286-3.978z"/>
                                                    </svg>
                                                <?php else: ?>
                                                    <svg class="h-5 w-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M11.049 2.927c.3-.921 1.603-.921 1.902 0l1.286 3.978a1 1 0 00.95.69h4.21c.969 0 1.371 1.24.588 1.81l-3.404 2.475a1 1 0 00-.364 1.118l1.285 3.978c.3.921-.755 1.688-1.54 1.118l-3.404-2.475a1 1 0 00-1.175 0l-3.404 2.475c-.785.57-1.84-.197-1.54-1.118l1.285-3.978a1 1 0 00-.364-1.118L2.227 9.405c-.783-.57-.38-1.81.588-1.81h4.21a1 1 0 00.95-.69l1.286-3.978z"/>
                                                    </svg>
                                                <?php endif; ?>
                                            </button>
                                            <!-- Delete Button -->
                                            <button type="button" title="Delete Photo" @click.prevent="window.deletePhoto(<?php echo e($photo['id']); ?>)" class="bg-red-600 text-white rounded-full p-1 opacity-0 group-hover:opacity-100 transition-opacity mt-1">
                                                <svg class="h-4 w-4" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12" /></svg>
                                            </button>
                                        </div>
                                    </div>
                                <?php endforeach; ?>
                            </div>
                        <?php endif; ?>
                        <!-- New Photos -->
                        <div class="mt-2">
                            <label class="form-label">Add New Photos</label>
                            <div class="space-y-4 mt-2 border-2 border-dashed border-gray-400 p-4 rounded-xl" x-data="photoUploader()">
                                <template x-for="(slot, index) in imageSlots" :key="slot.id">
                                    <div class="flex items-start gap-4 p-3 bg-gray-50 rounded-lg border-2 border-gray-300">
                                        <div class="flex-shrink-0 w-1/4">
                                            <img x-show="slot.previewUrl" :src="slot.previewUrl" class="h-24 w-full object-cover rounded-lg border-2 border-black bg-gray-100">
                                            <div x-show="!slot.previewUrl" class="h-24 w-full bg-gray-200 rounded-lg flex items-center justify-center text-gray-400 border-2 border-black">
                                                <svg class="h-8 w-8" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14" /></svg>
                                            </div>
                                        </div>
                                        <div class="flex-1 space-y-2">
                                            <input type="file" name="project_photos[]" :id="'photo-input-' + index" @change="handleFileSelect(index, $event)" accept="image/*" class="form-input text-base">
                                            <input type="text" name="photo_caption[]" x-model="slot.caption" class="form-input" placeholder="Optional caption">
                                            <select name="photo_type[]" x-model="slot.type" class="form-select">
                                                <option value="Progress">Progress</option>
                                                <option value="Before">Before</option>
                                                <option value="After">After</option>
                                            </select>
                                        </div>
                                        <div>
                                            <button type="button" @click.prevent="removeImageSlot(index)" title="Remove Image" class="text-gray-400 hover:text-red-600 transition-colors p-1">
                                                <svg class="h-6 w-6" fill="none" viewBox="0 0 24 24" stroke="currentColor"><path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" /></svg>
                                            </button>
                                        </div>
                                    </div>
                                </template>
                                <div class="mt-4">
                                    <template x-if="imageSlots.length < 5">
                                        <button type="button" @click.prevent="addImageSlot()" class="w-full text-center px-4 py-2 border-2 border-dashed border-gray-400 rounded-lg text-base font-medium text-gray-700 hover:border-brand-purple hover:text-brand-purple transition-colors">
                                            + Add Another Image
                                        </button>
                                    </template>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tab 3: Status & Financials -->
                <div x-show="activeTab === 3" class="space-y-10" x-cloak>
                    <div>
                        <h2 class="text-xl font-bold text-gray-900 mb-2">Status & Timeline</h2>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            <div>
                                <label for="status" class="form-label">Project Status <span class="text-red-500">*</span></label>
                                <select id="status" name="status" required class="form-select">
                                    <?php $statuses = ['Planned', 'Ongoing', 'Completed', 'Stalled']; ?>
                                    <?php foreach($statuses as $s): ?>
                                        <option value="<?php echo $s; ?>" <?php echo (isset($project['status']) && $project['status'] == $s) ? 'selected' : ''; ?>><?php echo $s; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                            <div>
                                <label for="manual_progress" class="form-label">Manual Progress (%)</label>
                                <input type="number" id="manual_progress" name="manual_progress" min="0" max="100" class="form-input" value="<?php echo e($project['manual_progress'] ?? ''); ?>" placeholder="e.g. 75">
                            </div>
                            <div>
                                <label for="start_date" class="form-label">Start Date</label>
                                <input type="date" id="start_date" name="start_date" class="form-input" value="<?php echo e($project['start_date'] ?? ''); ?>">
                            </div>
                            <div>
                                <label for="expected_completion_date" class="form-label">Expected Completion</label>
                                <input type="date" id="expected_completion_date" name="expected_completion_date" class="form-input" value="<?php echo e($project['expected_completion_date'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                    <div class="form-section-divider"></div>
                    <div>
                         <h2 class="text-xl font-bold text-gray-900 mb-2">Financials & Contractor</h2>
                         <div class="grid grid-cols-1 md:grid-cols-2 gap-8">
                            <div>
                                <label for="budget_allocated" class="form-label">Budget Allocated (KES)</label>
                                <input type="number" id="budget_allocated" name="budget_allocated" step="0.01" class="form-input" value="<?php echo e($project['budget_allocated'] ?? '0.00'); ?>">
                            </div>
                            <div>
                                <label for="budget_spent" class="form-label">Budget Spent (KES)</label>
                                <input type="number" id="budget_spent" name="budget_spent" step="0.01" class="form-input" value="<?php echo e($project['budget_spent'] ?? '0.00'); ?>">
                            </div>
                            <div class="md:col-span-2">
                                <label for="contractor_name" class="form-label">Contractor Name</label>
                                <input type="text" id="contractor_name" name="contractor_name" class="form-input" value="<?php echo e($project['contractor_name'] ?? ''); ?>">
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <!-- Global Navigation & Submit Button -->
            <div class="mt-8 pt-6 border-t-2 border-black flex flex-wrap gap-2 justify-between">
                <button type="button" x-show="activeTab > 1" @click="activeTab--" class="rounded-lg bg-gray-800 text-white font-bold py-3 px-6 hover:bg-opacity-90 transition">← Previous</button>
                <div x-show="activeTab === 1" class="w-full flex justify-end">
                    <button type="button" @click="activeTab++" class="rounded-lg bg-brand-purple text-white font-bold py-3 px-6 hover:bg-opacity-90 transition">Next →</button>
                </div>
                <button type="button" x-show="activeTab === 2" @click="activeTab++" class="rounded-lg bg-brand-purple text-white font-bold py-3 px-6 hover:bg-opacity-90 transition">Next →</button>
                <button type="submit" x-show="activeTab === 3" class="w-full rounded-lg border border-transparent bg-brand-orange py-4 px-6 text-base font-bold text-white shadow-lg hover:bg-opacity-90 focus:outline-none focus:ring-2 focus:ring-brand-purple focus:ring-offset-2">
                    <?php echo $is_editing ? 'Save Changes' : 'Create Project'; ?>
                </button>
            </div>
        </div>
    </form>
</main>

<script src="//unpkg.com/alpinejs" defer></script>
<script>
// Alpine.js for tab switching
function photoTabs() {
    return {
        activeTab: 1,
        initPhotos() {}
    }
}

// Star/favorite AJAX logic for featured photo
window.settingFeatured = false;
window.setAsFeaturedPhoto = function(photoId, projectId) {
    if (window.settingFeatured) return;
    window.settingFeatured = true;
    fetch('set_featured_photo.php', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ photo_id: photoId, project_id: projectId })
    })
    .then(r => r.json())
    .then(data => {
        window.settingFeatured = false;
        if (data.success) {
            location.reload(); // Reload to show new featured
        } else {
            alert('Failed to set featured photo: ' + data.message);
        }
    })
    .catch(() => {
        window.settingFeatured = false;
        alert('An error occurred. Try again.');
    });
}

// Photo uploader for new uploads
function photoUploader() {
    return {
        imageSlots: [{ id: 1, previewUrl: '', caption: '', type: 'Progress' }],
        nextSlotId: 2,
        maxImages: 5,
        addImageSlot() {
            if (this.imageSlots.length < this.maxImages) {
                this.imageSlots.push({ id: this.nextSlotId++, previewUrl: '', caption: '', type: 'Progress' });
            }
        },
        removeImageSlot(index) {
            if (this.imageSlots.length === 1) {
                this.imageSlots[index] = { id: 1, previewUrl: '', caption: '', type: 'Progress' };
                document.getElementById(`photo-input-${index}`).value = '';
                return;
            }
            this.imageSlots.splice(index, 1);
        },
        handleFileSelect(index, event) {
            const file = event.target.files[0];
            if (file && file.type.startsWith('image/')) {
                if (this.imageSlots[index].previewUrl) {
                    URL.revokeObjectURL(this.imageSlots[index].previewUrl);
                }
                this.imageSlots[index].previewUrl = URL.createObjectURL(file);
            }
        },
        deletePhoto(photoId) {
            if (!confirm('Are you sure you want to delete this photo? This cannot be undone.')) return;
            fetch('delete_photo.php', {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ id: photoId })
            }).then(r => r.json()).then(data => {
                if (data.success) {
                    document.getElementById('photo-' + photoId)?.remove();
                } else {
                    alert('Error: ' + data.message);
                }
            }).catch(e => alert('An error occurred.'));
        }
    }
}
</script>
<?php include 'includes/footer.php'; ?>
