<?php
// FILE: admin/manage_article.php
// PURPOSE: A form for creating and editing news articles with a rich text editor.

require_once 'functions.php';
require_once 'db.php';

// Security: Only Admins and SuperAdmins can access
require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- Initialize variables ---
$article = [
    'id' => null,
    'title' => '',
    'content' => '',
    'featured_image' => null,
    'status' => 'Draft',
    'project_id' => null,
];
$errors = [];
$page_title = "Write New Article";
$is_edit_mode = false;

// --- Fetch projects for the optional dropdown ---
$approved_projects = [];
try {
    $approved_projects = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC")->fetchAll();
} catch (PDOException $e) {
    $errors['db_load'] = "Could not fetch projects list.";
}


// --- Handle GET request (for editing) ---
if (isset($_GET['id']) && !empty(trim($_GET['id']))) {
    $is_edit_mode = true;
    $article_id = trim($_GET['id']);
    $page_title = "Edit Article";

    $stmt = $pdo->prepare("SELECT * FROM news_articles WHERE id = :id");
    $stmt->execute(['id' => $article_id]);
    $article = $stmt->fetch();

    if (!$article) {
        header("location: news_dashboard.php");
        exit;
    }
}

// --- Handle POST request (form submission for add/edit) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $article_id = $_POST['id'] ?? null;
    $article['title'] = trim($_POST['title']);
    $article['content'] = trim($_POST['content']);
    $article['status'] = $_POST['status'] ?? 'Draft';
    $article['project_id'] = empty($_POST['project_id']) ? null : $_POST['project_id'];

    // Validation
    if (empty($article['title'])) $errors['title'] = "Article title is required.";
    if (empty($article['content']) || strlen(strip_tags($article['content'])) < 10) $errors['content'] = "Article content must be at least 10 characters.";

    // --- Featured Image Upload Logic ---
    $new_image_filename = $_POST['existing_image'] ?? null;
    if (isset($_FILES['featured_image']) && $_FILES['featured_image']['error'] == UPLOAD_ERR_OK) {
        $image = $_FILES['featured_image'];
        $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/webp'];
        if (!in_array($image['type'], $allowed_types)) {
            $errors['featured_image'] = "Invalid file type. Only JPG, PNG, GIF, WEBP are allowed.";
        } elseif ($image['size'] > 5000000) { // 5MB limit
            $errors['featured_image'] = "File is too large. Maximum size is 5MB.";
        } else {
            $upload_dir = 'uploads/';
            if (!is_dir($upload_dir)) mkdir($upload_dir, 0755, true);
            
            $new_image_filename = 'news_' . uniqid() . '.' . strtolower(pathinfo($image['name'], PATHINFO_EXTENSION));
            $destination = $upload_dir . $new_image_filename;
            
            if (!move_uploaded_file($image['tmp_name'], $destination)) {
                $errors['featured_image'] = "Failed to move uploaded file.";
                $new_image_filename = $_POST['existing_image'] ?? null; // Revert to old image on failure
            }
        }
    }
    $article['featured_image'] = $new_image_filename;

    if (empty($errors)) {
        try {
            if ($article_id) {
                // UPDATE existing article
                $sql = "UPDATE news_articles SET title = :title, content = :content, status = :status, featured_image = :featured_image, project_id = :project_id, author_id = :author_id WHERE id = :id";
            } else {
                // INSERT new article
                $sql = "INSERT INTO news_articles (title, content, status, featured_image, project_id, author_id) VALUES (:title, :content, :status, :featured_image, :project_id, :author_id)";
            }
            
            $stmt = $pdo->prepare($sql);
            
            $params_to_execute = [
                'title' => $article['title'],
                'content' => $article['content'],
                'status' => $article['status'],
                'featured_image' => $article['featured_image'],
                'project_id' => $article['project_id'],
                'author_id' => $_SESSION['id']
            ];

            if ($article_id) {
                $params_to_execute['id'] = $article_id;
            }

            $stmt->execute($params_to_execute);

            $new_id = $article_id ?: $pdo->lastInsertId();
            log_action($pdo, $article_id ? 'Updated Article' : 'Created Article', 'News', $new_id);
            
            header("location: news_dashboard.php?success=saved");
            exit;

        } catch (PDOException $e) {
            $errors['db'] = "Database Error: " . $e->getMessage();
        }
    }
}

include 'includes/header.php';
?>
<!-- Include Quill Rich Text Editor from CDN -->
<link href="https://cdn.quilljs.com/1.3.6/quill.snow.css" rel="stylesheet">
<script src="https://cdn.quilljs.com/1.3.6/quill.js"></script>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900"><?php echo e($page_title); ?></h1>
    </div>
</header>
    
<form action="manage_article.php<?php if($is_edit_mode) echo '?id=' . e($article['id']); ?>" method="post" enctype="multipart/form-data">
    <input type="hidden" name="id" value="<?php echo e($article['id']); ?>">
    <input type="hidden" name="content" id="quill-content">

    <div class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8 grid grid-cols-1 lg:grid-cols-3 gap-6">
        <!-- Main Content Column -->
        <div class="lg:col-span-2 space-y-6">
            <div class="bg-white p-6 rounded-lg shadow">
                <label for="title" class="block text-sm font-medium text-gray-700">Article Title</label>
                <input type="text" name="title" id="title" value="<?php echo e($article['title']); ?>" required class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm px-4 py-3 text-lg">
                <?php if(isset($errors['title'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['title']; ?></p><?php endif; ?>
            </div>

            <div class="bg-white rounded-lg shadow">
                <div class="p-6">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Article Content</label>
                    <!-- Quill Editor Container -->
                    <div id="editor" class="h-96">
                        <?php echo $article['content']; // Output raw HTML content into the editor ?>
                    </div>
                     <?php if(isset($errors['content'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['content']; ?></p><?php endif; ?>
                </div>
            </div>
        </div>

        <!-- Right Sidebar Column -->
        <div class="space-y-6">
            <div class="bg-white p-6 rounded-lg shadow">
                <h3 class="text-lg font-medium text-gray-900">Publish</h3>
                <div class="mt-4 space-y-4">
                     <div>
                        <label for="status" class="block text-sm font-medium text-gray-700">Status</label>
                        <select id="status" name="status" class="mt-1 block w-full rounded-md border-gray-300 py-2 pl-3 pr-10 text-base focus:border-indigo-500 focus:outline-none focus:ring-indigo-500 sm:text-sm">
                            <option value="Draft" <?php if ($article['status'] == 'Draft') echo 'selected'; ?>>Draft</option>
                            <option value="Published" <?php if ($article['status'] == 'Published') echo 'selected'; ?>>Published</option>
                        </select>
                    </div>
                    <div class="flex items-center justify-between">
                         <a href="news_dashboard.php" class="text-sm text-gray-600 hover:underline">Cancel</a>
                        <button type="submit" class="inline-flex justify-center rounded-md border border-transparent bg-brand-orange py-2 px-4 text-sm font-medium text-white shadow-sm hover:bg-opacity-90">
                            <?php echo $is_edit_mode ? 'Update Article' : 'Save Article'; ?>
                        </button>
                    </div>
                </div>
            </div>
             <div class="bg-white p-6 rounded-lg shadow">
                <h3 class="text-lg font-medium text-gray-900">Featured Image</h3>
                <div class="mt-4">
                     <input type="hidden" name="existing_image" value="<?php echo e($article['featured_image']); ?>">
                    <?php if (!empty($article['featured_image']) && file_exists('uploads/' . $article['featured_image'])): ?>
                        <img src="uploads/<?php echo e($article['featured_image']); ?>" class="w-full h-auto rounded-md mb-4">
                    <?php endif; ?>
                    <label for="featured_image" class="block text-sm font-medium text-gray-700"><?php echo !empty($article['featured_image']) ? 'Replace Image' : 'Upload Image'; ?></label>
                    <input type="file" name="featured_image" id="featured_image" class="mt-1 block w-full text-sm text-slate-500 file:mr-4 file:py-2 file:px-4 file:rounded-full file:border-0 file:text-sm file:font-semibold file:bg-violet-50 file:text-brand-purple hover:file:bg-violet-100">
                     <?php if(isset($errors['featured_image'])): ?><p class="mt-1 text-xs text-red-600"><?php echo $errors['featured_image']; ?></p><?php endif; ?>
                </div>
            </div>
            <div class="bg-white p-6 rounded-lg shadow">
                <h3 class="text-lg font-medium text-gray-900">Link to Project (Optional)</h3>
                <div class="mt-4">
                     <label for="project_id" class="block text-sm font-medium text-gray-700">If this article is about a specific project, link it here.</label>
                     <select id="project_id" name="project_id" class="mt-1 block w-full rounded-md border-gray-300 py-2 pl-3 pr-10 text-base focus:border-indigo-500 focus:outline-none focus:ring-indigo-500 sm:text-sm">
                        <option value="">None</option>
                        <?php foreach($approved_projects as $proj): ?>
                           <option value="<?php echo e($proj['id']); ?>" <?php if (($article['project_id'] ?? null) == $proj['id']) echo 'selected'; ?>>
                               <?php echo e($proj['name']); ?>
                           </option>
                        <?php endforeach; ?>
                    </select>
                </div>
            </div>
        </div>
    </div>
</form>

<script>
// Initialize Quill editor
var quill = new Quill('#editor', {
    theme: 'snow',
    modules: {
        toolbar: [
            [{ 'header': [1, 2, 3, false] }],
            ['bold', 'italic', 'underline'],
            [{ 'list': 'ordered'}, { 'list': 'bullet' }],
            ['link', 'blockquote'],
            ['clean']
        ]
    }
});

// Sync Quill content with hidden input field before form submission
var form = document.querySelector('form');
form.onsubmit = function() {
  var content = document.querySelector('input[name=content]');
  content.value = quill.root.innerHTML;
};
</script>

<?php include 'includes/footer.php'; ?>
