<?php
// FILE: admin/logout.php
// PURPOSE: Logs the user out securely by destroying the session,
//          deleting the persistent auth token from the DB,
//          and clearing the "Remember Me" cookie.

// Initialize the session
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- "REMEMBER ME" COOKIE AND TOKEN INVALIDATION ---

// Check if a "remember_me" cookie is set
if (isset($_COOKIE['remember_me'])) {
    // Include the database connection
    require_once 'db.php';

    // Split the cookie into selector and validator
    list($selector, $validator) = explode(':', $_COOKIE['remember_me'], 2);

    if ($selector && $validator) {
        try {
            // Find and delete the token from the database using the selector
            // This ensures the token cannot be used again, even if the cookie was stolen.
            $stmt = $pdo->prepare("DELETE FROM auth_tokens WHERE selector = :selector");
            $stmt->execute(['selector' => $selector]);
        } catch (PDOException $e) {
            // Log the error for administrative review.
            // Don't show an error to the user, just proceed with logout.
            error_log("Logout token deletion failed: " . $e->getMessage());
        }
    }

    // Unset the cookie by setting its expiration date to the past
    setcookie('remember_me', '', time() - 3600, '/');
}


// --- SESSION DESTRUCTION ---

// Unset all of the session variables
$_SESSION = [];

// If you are using session cookies, it's a good practice to delete the session cookie as well.
// Note: This will destroy the session, and not just the session data!
if (ini_get("session.use_cookies")) {
    $params = session_get_cookie_params();
    setcookie(session_name(), '', time() - 42000,
        $params["path"], $params["domain"],
        $params["secure"], $params["httponly"]
    );
}

// Finally, destroy the session.
session_destroy();


// --- REDIRECT ---

// Redirect to the login page
header("location: login.php");
exit;
