<?php
// FILE: admin/forgot_password.php
// PURPOSE: Handles the first step of the password reset process using the styled mailer.

// --- DEPENDENCIES & HELPERS ---
require_once 'config.php';
require_once 'db.php';
require_once 'includes/mailer_template.php';

if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

function e($string) {
    return htmlspecialchars($string, ENT_QUOTES, 'UTF-8');
}

$errors = [];
$success_message = '';

// --- GENERATE CSRF TOKEN ---
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// --- FORM SUBMISSION LOGIC ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    if (!isset($_POST['csrf_token']) || !hash_equals($csrf_token, $_POST['csrf_token'])) {
        $errors['db'] = "Invalid request. Please try again.";
    } else {
        $email = trim($_POST['email']);
        if (empty($email)) {
            $errors['email'] = "Email address is required.";
        } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
            $errors['email'] = "Please enter a valid email address.";
        }

        if (empty($errors)) {
            try {
                // Check if an active user with this email exists
                $stmt = $pdo->prepare("SELECT id, name FROM users WHERE email = :email AND is_active = 1 AND deleted_at IS NULL");
                $stmt->execute(['email' => $email]);
                $user = $stmt->fetch();

                // IMPORTANT: For security, show the same success message whether the user exists or not.
                $success_message = "If an account with that email exists, a password reset link has been sent.";

                if ($user) {
                    // Generate a secure token for the user (this is what they see)
                    $token = bin2hex(random_bytes(50));
                    // Hash the token for database storage (this is what we check against)
                    $token_hash = hash('sha256', $token);
                    $expiry_time = date('Y-m-d H:i:s', time() + 3600); // Token is valid for 1 hour

                    // Delete any old tokens for this email to prevent multiple valid links
                    $pdo->prepare("DELETE FROM password_resets WHERE email = :email")->execute(['email' => $email]);

                    // Store the token HASH in the database
                    $sql = "INSERT INTO password_resets (email, token, expires_at) VALUES (:email, :token, :expires_at)";
                    $stmt = $pdo->prepare($sql);
                    $stmt->execute(['email' => $email, 'token' => $token_hash, 'expires_at' => $expiry_time]);

                    // --- Send the password reset email using the styled mailer ---
                    $reset_link = "https://genowa.co.ke/admin/reset_password.php?token=" . $token;
                    $subject = "Your Genowa Password Reset Request";
                    
                    $template_path = __DIR__ . '/includes/email_reset_template.html';
                    if (file_exists($template_path)) {
                        $email_body = file_get_contents($template_path);
                        $email_body = str_replace('{{user_name}}', e($user['name']), $email_body);
                        $email_body = str_replace('{{reset_link}}', $reset_link, $email_body);

                        $alt_body = "Hello " . e($user['name']) . ",\n\nWe received a request to reset your password. Please click the following link to proceed: " . $reset_link . "\n\nThis link is valid for one hour. If you did not request this, please ignore this email.\n\nRegards,\nThe Genowa Team";

                        if (!send_styled_email($email, $user['name'], $subject, $email_body, $alt_body, $config)) {
                            // The success message is still shown, but we log the error.
                            error_log("Failed to send password reset email to {$email}.");
                        }
                    } else {
                        error_log("Password reset email template not found at: " . $template_path);
                    }
                }
            } catch (PDOException $e) {
                error_log("Forgot Password PDOException: " . $e->getMessage());
                // Show a generic error to the user
                $errors['db'] = "A database error occurred. Please try again later.";
            }
        }
    }
}

include 'includes/header.php';
?>
<div class="flex items-center justify-center min-h-full px-4 py-12 sm:px-6 lg:px-8">
    <div class="w-full max-w-md space-y-8">
        <div>
            <a href="../index.php" class="flex items-center justify-center space-x-3">
                <img src="assets/hbc_logo.png" class="h-16 w-auto" alt="Homa Bay County Logo">
                <div class="h-10 w-px bg-gray-300"></div>
                <div>
                    <span class="font-extrabold text-2xl text-brand-blue">GENOWA</span>
                    <span class="block font-semibold text-sm text-brand-orange -mt-1">DELIVERY UNIT</span>
                </div>
            </a>
            <h2 class="mt-6 text-center text-3xl font-extrabold text-brand-blue">
                Reset Your Password
            </h2>
            <p class="mt-2 text-center text-sm text-gray-600">
                Enter your email address and we will send you a link to reset your password.
            </p>
        </div>
        <div class="bg-white p-8 rounded-xl shadow-lg space-y-6">
            <?php if ($success_message): ?>
                <div class="bg-green-100 border-l-4 border-green-500 text-green-700 p-4 rounded-md" role="alert">
                    <p><?php echo e($success_message); ?></p>
                </div>
            <?php endif; ?>

            <?php if (!empty($errors)): ?>
                <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded-md" role="alert">
                    <?php foreach ($errors as $error): ?>
                        <p><?php echo e($error); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            
            <form class="space-y-6" action="forgot_password.php" method="post">
                <input type="hidden" name="csrf_token" value="<?php echo e($csrf_token); ?>">
                <div>
                    <label for="email" class="sr-only">Email address</label>
                    <input id="email" name="email" type="email" autocomplete="email" required class="appearance-none relative block w-full px-3 py-3 border border-gray-300 placeholder-gray-500 text-gray-900 rounded-md focus:outline-none focus:ring-brand-purple focus:border-brand-purple sm:text-sm" placeholder="Enter your email address">
                </div>
                 <div>
                    <button type="submit" class="group relative w-full flex justify-center py-3 px-4 border border-transparent text-sm font-medium rounded-md text-white bg-brand-orange hover:bg-orange-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-brand-purple transition duration-300 ease-in-out">
                        Send Reset Link
                    </button>
                </div>
            </form>
             <p class="text-center text-sm text-gray-600">
                Remember your password? 
                <a href="login.php" class="font-medium text-brand-purple hover:text-brand-orange">
                    Sign in here
                </a>
            </p>
        </div>
    </div>
</div>
<?php include 'includes/footer.php'; ?>
