<?php
// FILE: admin/feedback_dashboard.php
// PURPOSE: To moderate all pending public feedback comments.

require_once 'functions.php';
require_once 'db.php';

// Security: Only Admins and SuperAdmins can access
require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- Handle Actions (Approve, Delete) ---
if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $feedback_id = $_POST['feedback_id'] ?? null;
    
    if ($feedback_id) {
        if (isset($_POST['approve'])) {
            $stmt = $pdo->prepare("UPDATE project_feedback SET is_approved = 1 WHERE id = :id");
            $stmt->execute(['id' => $feedback_id]);
            log_action($pdo, 'Approved Comment', 'Feedback', $feedback_id);
        }
        if (isset($_POST['delete'])) {
            $stmt = $pdo->prepare("DELETE FROM project_feedback WHERE id = :id");
            $stmt->execute(['id' => $feedback_id]);
            log_action($pdo, 'Deleted Comment', 'Feedback', $feedback_id);
        }
    }
    header("Location: feedback_dashboard.php"); // Redirect to prevent form resubmission
    exit;
}

// --- Setup Pagination ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 10;
$offset = ($page - 1) * $limit;

$feedback_items = [];
$total_feedback = 0;

try {
    // Get total count of pending comments
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM project_feedback WHERE is_approved = 0");
    $total_feedback = $count_stmt->fetchColumn();
    $total_pages = ceil($total_feedback / $limit);

    // Fetch pending comments for the current page
    $sql = "SELECT pf.*, p.name as project_name 
            FROM project_feedback pf
            JOIN projects p ON pf.project_id = p.id
            WHERE pf.is_approved = 0 
            ORDER BY pf.submitted_at DESC 
            LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $feedback_items = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching feedback: " . $e->getMessage();
}

include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">Public Feedback Moderation</h1>
        <p class="mt-1 text-sm text-gray-500">Total pending comments: <?php echo e(number_format($total_feedback)); ?></p>
    </div>
</header>
    
<div class="overflow-x-auto bg-white rounded-lg shadow mt-6">
    <?php if (isset($db_error)): ?>
        <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
    <?php elseif (empty($feedback_items)): ?>
        <div class="text-center p-12">
            <h3 class="mt-2 text-lg font-medium text-gray-900">No pending comments.</h3>
            <p class="mt-1 text-sm text-gray-500">All feedback has been moderated. Well done!</p>
        </div>
    <?php else: ?>
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Comment</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Project</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Submitted</th>
                    <th scope="col" class="relative px-6 py-3"><span class="sr-only">Actions</span></th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                <?php foreach ($feedback_items as $item): ?>
                    <tr>
                        <td class="px-6 py-4">
                            <div class="text-sm font-medium text-gray-900"><?php echo e($item['author_name']); ?></div>
                            <div class="text-sm text-gray-600 mt-1"><?php echo nl2br(e($item['comment'])); ?></div>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                           <a href="manage_project.php?id=<?php echo e($item['project_id']); ?>" class="hover:text-brand-purple"><?php echo e($item['project_name']); ?></a>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date('M j, Y, g:i a', strtotime($item['submitted_at'])); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <form method="POST" action="feedback_dashboard.php" class="inline-flex space-x-2">
                                <input type="hidden" name="feedback_id" value="<?php echo e($item['id']); ?>">
                                <button type="submit" name="approve" class="text-green-600 hover:text-green-900">Approve</button>
                                <button type="submit" name="delete" class="text-red-600 hover:text-red-900" onclick="return confirm('Are you sure you want to delete this comment?');">Delete</button>
                            </form>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    <div class="text-sm text-gray-700">Page <?php echo $page; ?> of <?php echo $total_pages; ?></div>
    <?php if ($page < $total_pages): ?>
        <a href="?page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
