<?php
// FILE: admin/dashboard.php
// PURPOSE: The main dashboard, now updated to work with the new sidebar layout.

require_once 'functions.php';
require_once 'db.php';

// Ensure user is logged in
require_login();

// --- Initialize variables ---
$pending_projects = [];
$approved_projects = [];
$stats = [
    'total' => 0, 'ongoing' => 0, 'completed' => 0, 
    'pending_users' => 0, 'planned' => 0, 'stalled' => 0,
    'pending_projects_count' => 0
];

$user_can_approve = is_super_admin() || is_admin() || is_approver();

try {
    // --- Fetch Projects Pending Approval ---
    if ($user_can_approve) {
        $pending_sql = "SELECT p.id, p.name as project_name, w.name as ward_name, u.name as created_by_name 
                        FROM projects p
                        JOIN wards w ON p.ward_id = w.id
                        JOIN users u ON p.created_by = u.id
                        WHERE p.is_approved = 0 
                        ORDER BY p.created_at DESC";
        $pending_stmt = $pdo->query($pending_sql);
        $pending_projects = $pending_stmt->fetchAll();
        $stats['pending_projects_count'] = count($pending_projects);
    }
    
    // --- Fetch Approved Projects (for main list) ---
    $approved_sql_base = "FROM projects p
                         JOIN wards w ON p.ward_id = w.id
                         JOIN constituencies c ON w.constituency_id = c.id
                         WHERE p.is_approved = 1";
    $params = [];

    // Filter by creator for Inputer role
    if(is_inputer()){
        $approved_sql_base .= " AND p.created_by = :user_id";
        $params['user_id'] = $_SESSION['id'];
    }

    $approved_sql = "SELECT p.id, p.name AS project_name, p.status, w.name AS ward_name, c.name AS constituency_name " . $approved_sql_base . " ORDER BY p.updated_at DESC";
    $approved_stmt = $pdo->prepare($approved_sql);
    $approved_stmt->execute($params);
    $approved_projects = $approved_stmt->fetchAll();


    // --- Fetch Stats (based on approved projects only) ---
    $stats_stmt = $pdo->query("SELECT status, COUNT(*) as count FROM projects WHERE is_approved = 1 GROUP BY status");
    $project_counts = $stats_stmt->fetchAll(PDO::FETCH_KEY_PAIR);
    $stats['ongoing'] = $project_counts['Ongoing'] ?? 0;
    $stats['completed'] = $project_counts['Completed'] ?? 0;
    $stats['planned'] = $project_counts['Planned'] ?? 0;
    $stats['stalled'] = $project_counts['Stalled'] ?? 0;
    $stats['total'] = array_sum($project_counts);

    // Fetch pending users count if SuperAdmin
    if (is_super_admin()) {
        $stmt = $pdo->query("SELECT COUNT(*) FROM users WHERE is_active = 0");
        $stats['pending_users'] = $stmt->fetchColumn();
    }

} catch (PDOException $e) {
    $db_error = "Error fetching dashboard data: " . $e->getMessage();
}

include 'includes/header.php';
?>
<!-- Include Chart.js library -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>

<!-- The page-specific header -->
<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900">Dashboard</h1>
    </div>
</header>
    
<!-- Main content starts here, inside the <main> tag opened in header.php -->
<div class="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-4 mb-6">
    <!-- KPI Cards -->
    <div class="overflow-hidden rounded-lg bg-white p-5 shadow"><p class="truncate text-sm font-medium text-gray-500">Approved Projects</p><p class="mt-1 text-3xl font-semibold tracking-tight text-gray-900"><?php echo e($stats['total']); ?></p></div>
    <div class="overflow-hidden rounded-lg bg-white p-5 shadow"><p class="truncate text-sm font-medium text-gray-500">Ongoing</p><p class="mt-1 text-3xl font-semibold tracking-tight text-brand-blue"><?php echo e($stats['ongoing']); ?></p></div>
    <div class="overflow-hidden rounded-lg bg-white p-5 shadow"><p class="truncate text-sm font-medium text-gray-500">Completed</p><p class="mt-1 text-3xl font-semibold tracking-tight text-green-600"><?php echo e($stats['completed']); ?></p></div>
    <?php if ($user_can_approve): ?>
        <a href="#pending" class="overflow-hidden rounded-lg bg-yellow-100 p-5 shadow hover:bg-yellow-200"><p class="truncate text-sm font-medium text-yellow-800">Pending Approval</p><p class="mt-1 text-3xl font-semibold tracking-tight text-yellow-900"><?php echo e($stats['pending_projects_count']); ?></p></a>
    <?php elseif(is_super_admin()): ?>
         <div class="overflow-hidden rounded-lg bg-white p-5 shadow"><p class="truncate text-sm font-medium text-gray-500">Pending Users</p><p class="mt-1 text-3xl font-semibold tracking-tight text-brand-orange"><?php echo e($stats['pending_users']); ?></p></div>
    <?php endif; ?>
</div>

<!-- Pending Approval Section -->
<?php if ($user_can_approve && !empty($pending_projects)): ?>
<div id="pending" class="mb-8">
    <div class="bg-yellow-50 border border-yellow-200 rounded-lg shadow-sm">
        <div class="p-4 border-b border-yellow-200"><h3 class="text-lg font-medium leading-6 text-yellow-900">Projects Pending Approval</h3></div>
        <ul role="list" class="divide-y divide-yellow-200">
            <?php foreach($pending_projects as $project): ?>
            <li class="flex items-center justify-between gap-x-6 px-4 py-3">
                <div class="min-w-0"><p class="text-sm font-semibold leading-6 text-gray-900"><?php echo e($project['project_name']); ?></p><p class="mt-1 truncate text-xs leading-5 text-gray-500">Submitted by <?php echo e($project['created_by_name']); ?> for <?php echo e($project['ward_name']); ?> Ward</p></div>
                <a href="manage_project.php?id=<?php echo e($project['id']); ?>" class="rounded-full bg-white px-2.5 py-1 text-xs font-semibold text-gray-900 shadow-sm ring-1 ring-inset ring-gray-300 hover:bg-gray-50">Review & Approve</a>
            </li>
            <?php endforeach; ?>
        </ul>
    </div>
</div>
<?php endif; ?>

<!-- Main Content Grid -->
<div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
    <div class="lg:col-span-2">
         <div class="overflow-x-auto bg-white rounded-lg shadow">
            <div class="p-4 border-b"><h3 class="text-lg font-medium leading-6 text-gray-900">Approved Projects List</h3></div>
            <?php if (isset($db_error)): ?>
                <div class="p-4" role="alert"><p><?php echo e($db_error); ?></p></div>
            <?php elseif (empty($approved_projects)): ?>
                <div class="text-center p-12"><h3 class="mt-2 text-lg font-medium text-gray-900">No approved projects found.</h3></div>
            <?php else: ?>
                <table class="min-w-full divide-y divide-gray-200">
                    <thead class="bg-gray-50">
                        <tr>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Project</th>
                            <th scope="col" class="hidden sm:table-cell px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Location</th>
                            <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                            <th scope="col" class="relative px-6 py-3"><span class="sr-only">Manage</span></th>
                        </tr>
                    </thead>
                    <tbody class="bg-white divide-y divide-gray-200">
                        <?php foreach ($approved_projects as $project): ?>
                            <tr>
                                <td class="px-6 py-4 whitespace-nowrap"><div class="text-sm font-medium text-gray-900"><?php echo e($project['project_name']); ?></div></td>
                                <td class="hidden sm:table-cell px-6 py-4 whitespace-nowrap">
                                    <div class="text-sm text-gray-900"><?php echo e($project['ward_name']); ?></div>
                                    <div class="text-sm text-gray-500"><?php echo e($project['constituency_name']); ?></div>
                                </td>
                                <td class="px-6 py-4 whitespace-nowrap"><span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full <?php switch ($project['status']) { case 'Completed': echo 'bg-green-100 text-green-800'; break; case 'Ongoing': echo 'bg-blue-100 text-blue-800'; break; case 'Planned': echo 'bg-yellow-100 text-yellow-800'; break; case 'Stalled': echo 'bg-red-100 text-red-800'; break; }?>"><?php echo e($project['status']); ?></span></td>
                                <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium"><a href="manage_project.php?id=<?php echo $project['id']; ?>" class="text-brand-purple hover:text-brand-orange">Manage</a></td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="bg-white rounded-lg shadow p-4">
         <h3 class="text-lg font-medium leading-6 text-gray-900 mb-4">Approved Projects by Status</h3>
        <div class="h-64">
             <canvas id="projectStatusChart"></canvas>
        </div>
    </div>
</div>

<script>
    document.addEventListener('DOMContentLoaded', function () {
        const ctx = document.getElementById('projectStatusChart');
        if (ctx) {
            new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: ['Ongoing', 'Completed', 'Planned', 'Stalled'],
                    datasets: [{
                        label: 'Project Status',
                        data: [
                            <?php echo e($stats['ongoing']); ?>, 
                            <?php echo e($stats['completed']); ?>,
                            <?php echo e($stats['planned']); ?>,
                            <?php echo e($stats['stalled']); ?>

                        ],
                        backgroundColor: ['#3B82F6', '#16A34A', '#FBBF24', '#DC2626'],
                        hoverOffset: 4
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: { legend: { position: 'bottom' } }
                }
            });
        }
    });
</script>

<?php 
// The footer now closes the main content area and the page
include 'includes/footer.php'; 
?>
