<?php
require_once 'functions.php';
require_once 'db.php';
require_login();

include 'includes/header.php';
?>
<!-- Chart.js & Tailwind (load from CDN if not present) -->
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script src="https://cdn.tailwindcss.com"></script>

<header class="bg-white shadow-sm">
  <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8 flex items-center justify-between">
    <h1 class="text-lg font-semibold leading-6 text-gray-900">Dashboard</h1>
    <div class="flex space-x-2">
      <a href="manage_project.php" class="rounded px-3 py-2 bg-brand-blue text-white font-semibold shadow hover:bg-blue-800 text-sm">+ Add Project</a>
      <button onclick="exportProjects()" class="rounded px-3 py-2 bg-brand-orange text-white font-semibold shadow hover:bg-orange-700 text-sm">Export</button>
      <a href="../index.php" target="_blank" class="rounded px-3 py-2 bg-brand-purple text-white font-semibold shadow hover:bg-purple-700 text-sm">Public Portal</a>
    </div>
  </div>
</header>

<div class="grid grid-cols-1 gap-6 sm:grid-cols-2 lg:grid-cols-4 mb-6 mt-2 px-4">
  <button onclick="filterStatus('All')" class="overflow-hidden rounded-lg bg-white p-5 shadow w-full transition hover:scale-105">
    <p class="truncate text-sm font-medium text-gray-500">All Approved</p>
    <p class="mt-1 text-3xl font-semibold tracking-tight text-gray-900" id="kpi-total">--</p>
  </button>
  <button onclick="filterStatus('Ongoing')" class="overflow-hidden rounded-lg bg-white p-5 shadow w-full transition hover:scale-105">
    <p class="truncate text-sm font-medium text-gray-500">Ongoing</p>
    <p class="mt-1 text-3xl font-semibold tracking-tight text-brand-blue" id="kpi-ongoing">--</p>
  </button>
  <button onclick="filterStatus('Completed')" class="overflow-hidden rounded-lg bg-white p-5 shadow w-full transition hover:scale-105">
    <p class="truncate text-sm font-medium text-gray-500">Completed</p>
    <p class="mt-1 text-3xl font-semibold tracking-tight text-green-600" id="kpi-completed">--</p>
  </button>
  <button onclick="filterStatus('Planned')" class="overflow-hidden rounded-lg bg-white p-5 shadow w-full transition hover:scale-105">
    <p class="truncate text-sm font-medium text-gray-500">Planned</p>
    <p class="mt-1 text-3xl font-semibold tracking-tight text-yellow-700" id="kpi-planned">--</p>
  </button>
</div>

<!-- Pending Approval -->
<div id="pending-section"></div>

<!-- Approved Projects List -->
<div class="mx-4 mb-8 bg-white rounded-lg shadow">
  <div class="p-4 flex flex-col md:flex-row md:items-center md:justify-between border-b">
    <h3 class="text-lg font-medium leading-6 text-gray-900">Approved Projects</h3>
    <form id="search-form" class="w-full md:w-1/3 mt-2 md:mt-0">
      <input type="search" name="search" id="search-input" class="block w-full rounded-lg border-gray-300 p-2 text-gray-800 focus:border-brand-blue focus:ring-1 focus:ring-brand-blue" placeholder="Search by name, ward, constituency, status...">
    </form>
  </div>
  <div id="table-loading" class="p-6 text-center text-brand-blue hidden">
    <svg class="animate-spin h-6 w-6 mx-auto mb-2" viewBox="0 0 24 24"><circle class="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" stroke-width="4"></circle><path class="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8v8z"></path></svg>
    <p>Loading projects...</p>
  </div>
  <div class="overflow-x-auto">
    <table id="projects-table" class="min-w-full divide-y divide-gray-200">
      <thead class="bg-gray-50">
        <tr>
          <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Project</th>
          <th class="hidden sm:table-cell px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Location</th>
          <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
          <th class="px-6 py-3 text-right text-xs font-medium text-gray-500 uppercase tracking-wider">Action</th>
        </tr>
      </thead>
      <tbody id="projects-body" class="bg-white divide-y divide-gray-200"></tbody>
    </table>
  </div>
  <div id="no-data" class="p-8 text-center text-gray-600 hidden">
    <svg class="mx-auto mb-2 h-10 w-10 text-gray-400" fill="none" viewBox="0 0 24 24"><path d="M15 12a3 3 0 01-6 0V7.5a3 3 0 016 0V12z" /></svg>
    <div class="font-semibold">No approved projects found.</div>
    <div class="text-sm text-gray-400">Try changing your filters or search term.</div>
  </div>
  <div id="pagination" class="p-4 text-center"></div>
</div>

<div class="mx-4 mb-8 grid grid-cols-1 md:grid-cols-2 gap-6">
  <!-- Status Chart -->
  <div class="bg-white rounded-lg shadow p-4 flex flex-col">
    <h3 class="text-lg font-medium leading-6 text-gray-900 mb-4">Approved Projects by Status</h3>
    <div class="h-64 flex-1">
      <canvas id="projectStatusChart"></canvas>
    </div>
  </div>
  <!-- Constituency Bar Chart -->
  <div class="bg-white rounded-lg shadow p-4 flex flex-col">
    <h3 class="text-lg font-medium leading-6 text-gray-900 mb-4">Projects by Constituency</h3>
    <div class="h-72 flex-1">
      <canvas id="projectsByConstituencyChart"></canvas>
    </div>
  </div>
</div>


<script>
const PAGE_SIZE = 10;
let statusFilter = 'All';
let searchTerm = '';
let currentPage = 1;

function statusLabelClass(status) {
  switch (status) {
    case 'Completed': return 'bg-green-100 text-green-800';
    case 'Ongoing': return 'bg-blue-100 text-blue-800';
    case 'Planned': return 'bg-yellow-100 text-yellow-800';
    case 'Stalled': return 'bg-red-100 text-red-800';
    default: return 'bg-gray-100 text-gray-800';
  }
}
function filterStatus(status) {
  statusFilter = status;
  currentPage = 1;
  fetchProjects();
}
function gotoPage(p) {
  currentPage = p;
  fetchProjects(p);
}
let searchTimeout = null;
document.getElementById('search-input').addEventListener('input', function() {
  searchTerm = this.value;
  currentPage = 1;
  clearTimeout(searchTimeout);
  searchTimeout = setTimeout(fetchProjects, 350);
});
function exportProjects() {
  window.open(`approved_projects_data.php?search=${encodeURIComponent(searchTerm)}&status=${encodeURIComponent(statusFilter)}&export=1`, '_blank');
}
function updateKPIs(kpis) {
  document.getElementById('kpi-total').textContent = kpis.total || '0';
  document.getElementById('kpi-ongoing').textContent = kpis.ongoing || '0';
  document.getElementById('kpi-completed').textContent = kpis.completed || '0';
  document.getElementById('kpi-planned').textContent = kpis.planned || '0';
}
let projectChart = null;
function updateChart(kpis) {
  const data = [kpis.ongoing || 0, kpis.completed || 0, kpis.planned || 0, kpis.stalled || 0];
  if (!projectChart) {
    const ctx = document.getElementById('projectStatusChart').getContext('2d');
    projectChart = new Chart(ctx, {
      type: 'doughnut',
      data: {
        labels: ['Ongoing', 'Completed', 'Planned', 'Stalled'],
        datasets: [{
          data: data,
          backgroundColor: ['#3B82F6', '#16A34A', '#FBBF24', '#DC2626'],
          hoverOffset: 4
        }]
      },
      options: {
        responsive: true, maintainAspectRatio: false,
        plugins: { legend: { position: 'bottom' } }
      }
    });
  } else {
    projectChart.data.datasets[0].data = data;
    projectChart.update();
  }
}
// Constituency Bar Chart
let constituencyChart = null;
function renderConstituencyChart(data) {
  const ctx = document.getElementById('projectsByConstituencyChart').getContext('2d');
  const labels = data.map(item => item.constituency);
  const totals = data.map(item => item.total);
  if (constituencyChart) {
    constituencyChart.data.labels = labels;
    constituencyChart.data.datasets[0].data = totals;
    constituencyChart.update();
  } else {
    constituencyChart = new Chart(ctx, {
      type: 'bar',
      data: {
        labels: labels,
        datasets: [{
          label: 'Projects',
          data: totals,
          borderWidth: 1
        }]
      },
      options: {
        indexAxis: 'y',
        responsive: true,
        maintainAspectRatio: false,
        plugins: { legend: { display: false } },
        scales: {
          x: { beginAtZero: true, title: { display: true, text: 'Number of Projects' } }
        }
      }
    });
  }
}
// Main fetch
function fetchProjects(page = 1) {
  document.getElementById('table-loading').classList.remove('hidden');
  document.getElementById('projects-body').innerHTML = '';
  document.getElementById('no-data').classList.add('hidden');
  document.getElementById('pagination').innerHTML = '';
  fetch(`approved_projects_data.php?search=${encodeURIComponent(searchTerm)}&status=${encodeURIComponent(statusFilter)}&page=${page}&page_size=${PAGE_SIZE}`)
    .then(r => r.json())
    .then(res => {
      document.getElementById('table-loading').classList.add('hidden');
      updateKPIs(res.kpis || {});
      updateChart(res.kpis || {});
      renderConstituencyChart(res.by_constituency || []);
      if (!res.projects || res.projects.length === 0) {
        document.getElementById('no-data').classList.remove('hidden');
        return;
      }
      let html = '';
      res.projects.forEach(project => {
        html += `
        <tr>
          <td class="px-6 py-4 whitespace-nowrap">
            <div class="text-sm font-medium text-gray-900">${project.project_name}</div>
          </td>
          <td class="hidden sm:table-cell px-6 py-4 whitespace-nowrap">
            <div class="text-sm text-gray-900">${project.ward_name}</div>
            <div class="text-sm text-gray-500">${project.constituency_name}</div>
          </td>
          <td class="px-6 py-4 whitespace-nowrap">
            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${statusLabelClass(project.status)}">${project.status}</span>
          </td>
          <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
            <a href="manage_project.php?id=${project.id}" class="text-brand-purple hover:text-brand-orange">Manage</a>
          </td>
        </tr>
        `;
      });
      document.getElementById('projects-body').innerHTML = html;
      let pages = Math.ceil(res.total / PAGE_SIZE);
      let pagHtml = '';
      if (pages > 1) {
        for (let i = 1; i <= pages; i++) {
          pagHtml += `<button onclick="gotoPage(${i})" class="inline-block px-3 py-1 mx-1 rounded ${i === page ? 'bg-brand-blue text-white' : 'bg-gray-100 text-gray-800 hover:bg-brand-purple hover:text-white'}">${i}</button>`;
        }
        document.getElementById('pagination').innerHTML = pagHtml;
      }
    })
    .catch(() => {
      document.getElementById('table-loading').classList.add('hidden');
      document.getElementById('no-data').classList.remove('hidden');
    });
}
// Pending Approval
function fetchPending() {
  fetch('approved_projects_data.php?pending=1')
    .then(r => r.text()).then(html => {
      document.getElementById('pending-section').innerHTML = html;
    });
}
fetchProjects();
fetchPending();
</script>
<?php include 'includes/footer.php'; ?>
