<?php
// FILE: admin/approvals.php
// PURPOSE: Page for users with approval permissions to review and manage projects.
// VERSION: 2.0 - Added modal for rejection comments and enhanced display for approver/creator.

require_once 'functions.php';
require_once 'db.php';
require_login();

// --- Permissions Check ---
if (!is_approver() && !is_super_admin()) {
    $_SESSION['flash_message'] = "Error: You do not have permission to access the approvals page.";
    header("location: dashboard.php");
    exit;
}

include 'includes/header.php';

// --- Fetch projects for all tabs ---
try {
    // Pending Projects (is_approved = 0)
    $pending_stmt = $pdo->prepare("
        SELECT p.id, p.name, u.name as creator_name, w.name as ward_name, p.created_at
        FROM projects p 
        JOIN users u ON p.created_by = u.id
        JOIN wards w ON p.ward_id = w.id
        WHERE p.is_approved = 0 
        ORDER BY p.created_at DESC
    ");
    $pending_stmt->execute();
    $pending_projects = $pending_stmt->fetchAll();

    // Approved Projects (is_approved = 1)
    $approved_stmt = $pdo->prepare("
        SELECT p.id, p.name, u_creator.name as creator_name, u_approver.name as approver_name, w.name as ward_name, p.approved_at
        FROM projects p 
        JOIN users u_creator ON p.created_by = u_creator.id
        LEFT JOIN users u_approver ON p.approved_by = u_approver.id
        JOIN wards w ON p.ward_id = w.id
        WHERE p.is_approved = 1 
        ORDER BY p.approved_at DESC
        LIMIT 20
    ");
    $approved_stmt->execute();
    $approved_projects = $approved_stmt->fetchAll();

    // Changes Requested Projects (is_approved = 2)
    $changes_stmt = $pdo->prepare("
        SELECT p.*, u.name as creator_name, w.name as ward_name 
        FROM projects p 
        JOIN users u ON p.created_by = u.id
        JOIN wards w ON p.ward_id = w.id
        WHERE p.is_approved = 2
        ORDER BY p.updated_at DESC
    ");
    $changes_stmt->execute();
    $changes_projects = $changes_stmt->fetchAll();

} catch(PDOException $e) {
    $error_message = "Database error fetching projects: " . $e->getMessage();
}

?>

<header class="bg-white shadow-sm">
  <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
    <h1 class="text-xl font-bold leading-6 text-gray-900">Project Approval Center</h1>
    <p class="mt-1 text-sm text-gray-500">Review and manage the status of submitted projects.</p>
  </div>
</header>

<main class="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8 py-8" x-data="{ 
    tab: 'pending', 
    showCommentModal: false, 
    modalProjectId: null, 
    modalProjectName: '', 
    rejectionComment: '' 
}">
    <?php if (isset($_SESSION['flash_message'])): ?>
    <div class="mb-6 p-4 rounded-md <?php echo strpos($_SESSION['flash_message'], 'Error') !== false ? 'bg-red-100 text-red-700' : 'bg-green-100 text-green-700'; ?>">
        <?php echo e($_SESSION['flash_message']); unset($_SESSION['flash_message']); ?>
    </div>
    <?php endif; ?>
    <?php if (isset($error_message)): ?>
        <div class="mb-6 p-4 rounded-md bg-red-100 text-red-700"><?php echo e($error_message); ?></div>
    <?php endif; ?>

    <div>
        <div class="border-b border-gray-200">
            <nav class="-mb-px flex space-x-8" aria-label="Tabs">
                <button @click="tab = 'pending'" :class="{ 'border-brand-purple text-brand-purple': tab === 'pending', 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300': tab !== 'pending' }" class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                    Pending Approval <span class="ml-2 px-2.5 py-1 rounded-full text-xs font-medium bg-yellow-200 text-yellow-800"><?php echo count($pending_projects); ?></span>
                </button>
                <button @click="tab = 'approved'" :class="{ 'border-brand-purple text-brand-purple': tab === 'approved', 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300': tab !== 'approved' }" class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                    Recently Approved
                </button>
                <button @click="tab = 'changes'" :class="{ 'border-brand-purple text-brand-purple': tab === 'changes', 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300': tab !== 'changes' }" class="whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm">
                    Changes Requested <span class="ml-2 px-2.5 py-1 rounded-full text-xs font-medium bg-red-200 text-red-800"><?php echo count($changes_projects); ?></span>
                </button>
            </nav>
        </div>

        <div class="mt-6">
            <div x-show="tab === 'pending'">
                <?php if(empty($pending_projects)): ?>
                    <p class="text-center text-gray-500 py-12">There are no projects currently awaiting approval. Great job!</p>
                <?php else: ?>
                    <div class="space-y-4">
                        <?php foreach($pending_projects as $project): ?>
                            <div class="bg-white p-4 rounded-lg shadow flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                                <div class="flex-grow">
                                    <p class="font-bold text-brand-blue"><?php echo e($project['name']); ?></p>
                                    <p class="text-sm text-gray-600">Location: <span class="font-medium"><?php echo e($project['ward_name']); ?></span></p>
                                    <p class="text-sm text-gray-500">Submitted by <?php echo e($project['creator_name']); ?> on <?php echo date('M j, Y', strtotime($project['created_at'])); ?></p>
                                </div>
                                <div class="flex-shrink-0 flex items-center gap-2">
                                    <a href="manage_project.php?id=<?php echo e($project['id']); ?>" class="px-4 py-2 text-sm font-medium text-gray-700 bg-white border border-gray-300 rounded-md hover:bg-gray-50">View Details</a>
                                    <form action="handle_approval.php" method="POST" class="inline-block" onsubmit="return confirm('Are you sure you want to approve this project?');">
                                        <input type="hidden" name="project_id" value="<?php echo e($project['id']); ?>">
                                        <input type="hidden" name="action" value="approve">
                                        <button type="submit" class="px-4 py-2 text-sm font-medium text-white bg-green-600 border border-transparent rounded-md hover:bg-green-700">Approve</button>
                                    </form>
                                    <button @click="showCommentModal = true; modalProjectId = <?php echo e($project['id']); ?>; modalProjectName = '<?php echo e(addslashes($project['name'])); ?>';" type="button" class="px-4 py-2 text-sm font-medium text-white bg-red-600 border border-transparent rounded-md hover:bg-red-700">Request Changes</button>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>

            <div x-show="tab === 'approved'" style="display: none;">
                 <?php if(empty($approved_projects)): ?>
                    <p class="text-center text-gray-500 py-12">No projects have been approved yet.</p>
                <?php else: ?>
                    <div class="space-y-4">
                        <?php foreach($approved_projects as $project): ?>
                             <div class="bg-white p-4 rounded-lg shadow flex flex-col sm:flex-row items-start sm:items-center justify-between gap-4">
                                <div>
                                    <p class="font-bold text-gray-800"><?php echo e($project['name']); ?></p>
                                    <p class="text-sm text-gray-500">
                                        <span class="font-medium text-brand-blue">Input by:</span> <?php echo e($project['creator_name']); ?> | 
                                        <span class="font-medium text-brand-orange">Approved by:</span> <?php echo e($project['approver_name']); ?> on <?php echo date('M j, Y', strtotime($project['approved_at'])); ?>
                                    </p>
                                </div>
                                <a href="manage_project.php?id=<?php echo e($project['id']); ?>" class="flex-shrink-0 px-4 py-2 text-sm font-medium text-brand-purple hover:text-brand-orange">View</a>
                             </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
            
            <div x-show="tab === 'changes'" style="display: none;">
                 <?php if(empty($changes_projects)): ?>
                    <p class="text-center text-gray-500 py-12">No projects are currently marked for changes.</p>
                <?php else: ?>
                    <div class="space-y-4">
                        <?php foreach($changes_projects as $project): ?>
                            <div class="bg-white p-4 rounded-lg shadow flex items-center justify-between">
                                <div>
                                    <p class="font-bold text-gray-800"><?php echo e($project['name']); ?></p>
                                    <p class="text-sm text-gray-500">Status updated on <?php echo date('M j, Y', strtotime($project['updated_at'])); ?></p>
                                </div>
                                <a href="manage_project.php?id=<?php echo e($project['id']); ?>" class="px-4 py-2 text-sm font-medium text-brand-purple hover:text-brand-orange">View & Edit</a>
                             </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <div x-show="showCommentModal" class="fixed inset-0 z-50 overflow-y-auto" aria-labelledby="modal-title" role="dialog" aria-modal="true" style="display: none;">
        <div class="flex items-end justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
            <div x-show="showCommentModal" x-transition:enter="ease-out duration-300" x-transition:enter-start="opacity-0" x-transition:enter-end="opacity-100" x-transition:leave="ease-in duration-200" x-transition:leave-start="opacity-100" x-transition:leave-end="opacity-0" class="fixed inset-0 bg-gray-500 bg-opacity-75 transition-opacity" @click="showCommentModal = false"></div>
            <span class="hidden sm:inline-block sm:align-middle sm:h-screen" aria-hidden="true">&#8203;</span>
            <div x-show="showCommentModal" x-transition:enter="ease-out duration-300" x-transition:enter-start="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95" x-transition:enter-end="opacity-100 translate-y-0 sm:scale-100" x-transition:leave="ease-in duration-200" x-transition:leave-start="opacity-100 translate-y-0 sm:scale-100" x-transition:leave-end="opacity-0 translate-y-4 sm:translate-y-0 sm:scale-95" class="inline-block align-bottom bg-white rounded-lg text-left overflow-hidden shadow-xl transform transition-all sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                <form action="handle_approval.php" method="POST">
                    <div class="bg-white px-4 pt-5 pb-4 sm:p-6 sm:pb-4">
                        <h3 class="text-lg leading-6 font-medium text-gray-900" id="modal-title">Request Changes</h3>
                        <p class="text-sm text-gray-600 mt-1">For project: <strong x-text="modalProjectName"></strong></p>
                        <div class="mt-4">
                            <label for="comment" class="block text-sm font-medium text-gray-700">Please provide a comment explaining the changes required:</label>
                            <textarea x-model="rejectionComment" name="comment" id="comment" rows="4" required class="mt-1 block w-full rounded-md border-gray-300 shadow-sm focus:border-brand-purple focus:ring-brand-purple"></textarea>
                        </div>
                    </div>
                    <div class="bg-gray-50 px-4 py-3 sm:px-6 sm:flex sm:flex-row-reverse">
                        <input type="hidden" name="project_id" :value="modalProjectId">
                        <input type="hidden" name="action" value="request_changes">
                        <button type="submit" class="w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 bg-red-600 text-base font-medium text-white hover:bg-red-700 sm:ml-3 sm:w-auto sm:text-sm">Submit Request</button>
                        <button type="button" @click="showCommentModal = false; rejectionComment = '';" class="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 shadow-sm px-4 py-2 bg-white text-base font-medium text-gray-700 hover:bg-gray-50 sm:mt-0 sm:w-auto sm:text-sm">Cancel</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</main>

<script src="//unpkg.com/alpinejs" defer></script>

<?php include 'includes/footer.php'; ?>