<?php
// FILE: public_html/index.php
// PURPOSE: The main landing page for the public Genowa Delivery Unit website.
// --- IMPLEMENTED IMPROVEMENTS ---
// 1. Performance: Replaced correlated subqueries with a more efficient LEFT JOIN pattern to prevent N+1 query issues.
// 2. Logic: Clarified the progress bar label to explicitly state when it's based on timeline.
// 3. Reliability: Switched image fallbacks to a local asset to remove external dependencies.
// 4. Feature: Added a "Citizen Journalism" call-to-action section.

require_once 'db_connect.php';
require_once 'admin/functions.php';

// --- Fetch data for the page ---
$stats = [
    'total_projects' => 0, 'completed_projects' => 0,
    'ongoing_projects' => 0, 'total_investment' => 0
];
$latest_completed_projects = [];
$ongoing_projects = [];
$my_constituency_projects = [];
$my_constituency_name = '';
$latest_news = [];

// --- Check for "My Constituency" cookie ---
$my_constituency_id = $_COOKIE['my_constituency_id'] ?? null;

try {
    // --- Fetch projects for the user's selected constituency ---
    if ($my_constituency_id) {
        $constituency_name_stmt = $pdo->prepare("SELECT name FROM constituencies WHERE id = :id");
        $constituency_name_stmt->execute(['id' => $my_constituency_id]);
        $my_constituency_name = $constituency_name_stmt->fetchColumn();

        if($my_constituency_name) {
            // IMPROVEMENT: Replaced correlated subquery with efficient LEFT JOIN
            $my_constituency_stmt = $pdo->prepare("
                SELECT
                    p.id, p.name, p.status, p.description, w.name as ward_name,
                    pi.image_url as project_image
                FROM projects p
                JOIN wards w ON p.ward_id = w.id
                LEFT JOIN (
                    SELECT project_id, image_url,
                           ROW_NUMBER() OVER(PARTITION BY project_id ORDER BY id DESC) as rn
                    FROM project_photos
                ) pi ON p.id = pi.project_id AND pi.rn = 1
                WHERE w.constituency_id = :constituency_id AND p.is_approved = 1
                ORDER BY w.name, p.status
                LIMIT 8
            ");
            $my_constituency_stmt->execute(['constituency_id' => $my_constituency_id]);
            $my_constituency_projects = $my_constituency_stmt->fetchAll();
        }
    }


    // Fetch statistics
    $stats_stmt = $pdo->query("SELECT COUNT(*) as total_projects, SUM(CASE WHEN status = 'Completed' THEN 1 ELSE 0 END) as completed_projects, SUM(CASE WHEN status = 'Ongoing' THEN 1 ELSE 0 END) as ongoing_projects, SUM(budget_allocated) as total_investment FROM projects WHERE is_approved = 1");
    $fetched_stats = $stats_stmt->fetch();
    if ($fetched_stats) { $stats = $fetched_stats; }

    // Fetch Ongoing Projects
    // IMPROVEMENT: Replaced correlated subquery with efficient LEFT JOIN
    $ongoing_stmt = $pdo->query("
        SELECT
            p.id, p.name, p.description, p.start_date, p.expected_completion_date, p.manual_progress, p.status, w.name as ward_name,
            pi.image_url as project_image
        FROM projects p
        JOIN wards w ON p.ward_id = w.id
        LEFT JOIN (
            SELECT project_id, image_url,
                   ROW_NUMBER() OVER(PARTITION BY project_id ORDER BY id DESC) as rn
            FROM project_photos
            WHERE photo_type = 'Progress'
        ) pi ON p.id = pi.project_id AND pi.rn = 1
        WHERE p.status = 'Ongoing' AND p.is_approved = 1
        ORDER BY p.expected_completion_date ASC
        LIMIT 6
    ");
    $ongoing_projects_raw = $ongoing_stmt->fetchAll();
    
    $today = new DateTime();
    foreach ($ongoing_projects_raw as $project) {
        $completion_percentage = 0;
        if (isset($project['manual_progress']) && is_numeric($project['manual_progress'])) {
            $completion_percentage = (int)$project['manual_progress'];
        } elseif ($project['start_date'] && $project['expected_completion_date']) {
            $start_date = new DateTime($project['start_date']);
            $expected_date = new DateTime($project['expected_completion_date']);
            if ($start_date <= $expected_date && $today >= $start_date) {
                $total_duration = $start_date->diff($expected_date)->days;
                $elapsed_duration = $start_date->diff($today)->days;
                if ($total_duration > 0) {
                    $percentage = min(100, ($elapsed_duration / $total_duration) * 100);
                    $completion_percentage = floor($percentage);
                } else { $percentage = 100; }
            }
        }
        $project['completion_percentage'] = $completion_percentage;
        $ongoing_projects[] = $project;
    }

    // Fetch Completed Projects
    // IMPROVEMENT: Replaced correlated subquery with efficient LEFT JOIN
    $projects_stmt = $pdo->query("
        SELECT
            p.id, p.name, p.description, w.name AS ward_name,
            pi.image_url as project_image
        FROM projects p
        JOIN wards w ON p.ward_id = w.id
        LEFT JOIN (
            SELECT project_id, image_url,
                   ROW_NUMBER() OVER(PARTITION BY project_id ORDER BY id DESC) as rn
            FROM project_photos
            WHERE photo_type IN ('After', 'Progress')
        ) pi ON p.id = pi.project_id AND pi.rn = 1
        WHERE p.status = 'Completed' AND p.is_approved = 1
        ORDER BY p.actual_completion_date DESC, p.id DESC
        LIMIT 6
    ");
    $latest_completed_projects = $projects_stmt->fetchAll();

    // Fetch latest 3 published news articles
    $news_stmt = $pdo->query("SELECT id, title, slug, content, featured_image, created_at FROM news_articles WHERE status = 'Published' ORDER BY created_at DESC LIMIT 3");
    $latest_news = $news_stmt->fetchAll();


} catch (PDOException $e) {
    // It's good practice to log the actual error for developers while showing a generic message to users.
    // error_log($e->getMessage()); 
    $db_error_message = "The website is currently experiencing technical difficulties. Please try again later.";
}

include 'includes/header.php';
?>

<div class="relative bg-brand-blue text-white overflow-hidden">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-24 md:py-32 text-center">
        <h1 class="text-4xl md:text-6xl font-black tracking-tight">Showcasing Homa Bay's</h1>
        <p class="mt-4 text-4xl md:text-6xl font-black tracking-tight text-brand-orange">Endless Potential</p>
        <p class="mt-6 max-w-2xl mx-auto text-lg md:text-xl text-gray-300">
            Transparently tracking the progress of development projects across all 40 wards of Homa Bay County.
        </p>
        <div class="mt-10">
            <a href="projects.php" class="bg-brand-orange text-white font-bold py-4 px-8 rounded-full text-lg hover:bg-opacity-90 transition duration-300 transform hover:scale-105">
                Explore All Projects
            </a>
        </div>
    </div>
</div>


<?php if (!empty($my_constituency_projects)): ?>
<div class="bg-yellow-50 dark:bg-yellow-900/20 py-16">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 fade-in-section">
        <div class="text-center mb-12">
            <h2 class="text-3xl md:text-4xl font-extrabold text-brand-blue dark:text-white">Projects in Your Constituency: <?php echo e($my_constituency_name); ?></h2>
            <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300">
                Here's a look at the development happening in the wards within your selected constituency.
            </p>
        </div>
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-8">
            <?php foreach ($my_constituency_projects as $project): ?>
                <a href="project.php?id=<?php echo e($project['id']); ?>" class="group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                    <div class="relative">
                        <img src="admin/uploads/<?php echo e($project['project_image'] ?? 'assets/images/placeholder.jpg'); ?>" alt="Image for <?php echo e($project['name']); ?>" class="h-48 w-full object-cover" onerror="this.onerror=null;this.src='assets/images/placeholder.jpg';">
                        <div class="absolute top-2 right-2 px-2 py-1 text-xs font-semibold rounded-full text-white <?php echo ($project['status'] == 'Completed') ? 'bg-green-600' : 'bg-blue-600'; ?>"><?php echo e($project['status']); ?></div>
                    </div>
                    <div class="p-4">
                        <p class="text-xs font-bold text-brand-orange uppercase"><?php echo e($project['ward_name']); ?> Ward</p>
                        <h3 class="mt-1 text-md font-bold text-brand-blue dark:text-white group-hover:text-brand-purple truncate"><?php echo e($project['name']); ?></h3>
                        <p class="mt-1 text-sm text-gray-600 dark:text-gray-400 line-clamp-2 h-10"><?php echo e($project['description']); ?></p>
                    </div>
                </a>
            <?php endforeach; ?>
        </div>
    </div>
</div>
<?php endif; ?>


<div class="bg-white dark:bg-dark-card py-12">
     <div class="container mx-auto px-4 sm:px-6 lg:px-8">
        <div class="grid grid-cols-1 md:grid-cols-3 gap-8 text-center fade-in-section">
            <div class="p-4"><p class="text-5xl font-extrabold text-brand-purple" data-countup="<?php echo e($stats['total_projects']); ?>">0</p><p class="mt-2 text-lg font-semibold text-gray-600 dark:text-gray-300">Approved Projects</p></div>
            <div class="p-4"><p class="text-5xl font-extrabold text-brand-purple" data-countup="<?php echo e($stats['completed_projects']); ?>">0</p><p class="mt-2 text-lg font-semibold text-gray-600 dark:text-gray-300">Completed Projects</p></div>
            <div class="p-4"><p class="text-5xl font-extrabold text-brand-purple"><span class="text-3xl align-top">KES</span> <span data-countup="<?php echo number_format($stats['total_investment'] / 1000000, 1); ?>">0</span><span class="text-3xl">M</span></p><p class="mt-2 text-lg font-semibold text-gray-600 dark:text-gray-300">Total Investment</p></div>
        </div>
    </div>
</div>

<div class="bg-white dark:bg-dark-card py-16">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8">
        <div class="text-center mb-12 fade-in-section">
            <h2 class="text-3xl md:text-4xl font-extrabold text-brand-blue dark:text-white">Recently Completed Projects</h2>
            <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300">Celebrating the successful delivery of projects that directly impact our citizens.</p>
        </div>
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
            <?php foreach ($latest_completed_projects as $project): ?>
                <a href="project.php?id=<?php echo e($project['id']); ?>" class="fade-in-section group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                    <div class="relative"><img src="admin/uploads/<?php echo e($project['project_image'] ?? 'assets/images/placeholder.jpg'); ?>" alt="Project image for <?php echo e($project['name']); ?>" class="h-56 w-full object-cover" onerror="this.onerror=null;this.src='assets/images/placeholder.jpg';"><div class="absolute inset-0 bg-black bg-opacity-20 group-hover:bg-opacity-40 transition-all duration-300"></div><div class="absolute top-3 right-3 px-2 py-1 text-xs font-semibold rounded-full text-white bg-green-600">Completed</div></div>
                    <div class="p-6"><p class="text-sm font-semibold text-brand-orange uppercase"><?php echo e($project['ward_name']); ?> Ward</p><h3 class="mt-2 text-xl font-bold text-brand-blue dark:text-white group-hover:text-brand-purple transition-colors duration-300"><?php echo e($project['name']); ?></h3><p class="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-2 h-10"><?php echo e($project['description']); ?></p><div class="mt-4"><div class="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5"><div class="bg-green-600 h-2.5 rounded-full" style="width: 100%"></div></div><div class="text-right text-sm font-semibold mt-1 text-gray-600 dark:text-gray-400">100% Complete</div></div></div>
                </a>
            <?php endforeach; ?>
        </div>
        <div class="mt-12 text-center">
            <a href="projects.php?status[]=Completed" class="bg-brand-orange text-white font-bold py-3 px-6 rounded-full text-base hover:bg-opacity-90">View All Completed Projects</a>
        </div>
    </div>
</div>

<div class="py-16 dark:bg-dark-bg">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8">
         <div class="text-center mb-12 fade-in-section">
             <h2 class="text-3xl md:text-4xl font-extrabold text-brand-blue dark:text-white">Ongoing Projects</h2>
             <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300">A real-time look at projects currently under development.</p>
        </div>
        <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
            <?php foreach ($ongoing_projects as $project): ?>
                <a href="project.php?id=<?php echo e($project['id']); ?>" class="fade-in-section group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                     <div class="relative"><img src="admin/uploads/<?php echo e($project['project_image'] ?? 'assets/images/placeholder.jpg'); ?>" alt="Progress image for <?php echo e($project['name']); ?>" class="h-56 w-full object-cover" onerror="this.onerror=null;this.src='assets/images/placeholder.jpg';"><div class="absolute inset-0 bg-black bg-opacity-20 group-hover:bg-opacity-40 transition-all duration-300"></div><div class="absolute top-3 right-3 px-2 py-1 text-xs font-semibold rounded-full text-white bg-blue-600">Ongoing</div></div>
                    <div class="p-6"><p class="text-sm font-semibold text-brand-orange uppercase"><?php echo e($project['ward_name']); ?> Ward</p><h3 class="mt-2 text-xl font-bold text-brand-blue dark:text-white group-hover:text-brand-purple transition-colors duration-300 truncate"><?php echo e($project['name']); ?></h3><p class="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-2 h-10"><?php echo e($project['description']); ?></p><div class="mt-4"><div class="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5"><div class="bg-brand-orange h-2.5 rounded-full" style="width: <?php echo e($project['completion_percentage']); ?>%"></div></div>
                    <div class="text-right text-sm font-semibold mt-1 text-gray-600 dark:text-gray-400">
                        <?php echo e($project['completion_percentage']); ?>% Complete
                        <?php if (!isset($project['manual_progress']) || !is_numeric($project['manual_progress'])): ?>
                            <span class="font-normal">(by timeline)</span>
                        <?php endif; ?>
                    </div>
                    </div></div>
                </a>
            <?php endforeach; ?>
        </div>
        <div class="mt-12 text-center">
            <a href="projects.php?status[]=Ongoing" class="bg-brand-orange text-white font-bold py-3 px-6 rounded-full text-base hover:bg-opacity-90">View All Ongoing Projects</a>
        </div>
    </div>
</div>

<div class="py-16">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8">
        <div class="text-center mb-12 fade-in-section">
            <h2 class="text-3xl md:text-4xl font-extrabold text-brand-blue dark:text-white">News & Updates</h2>
            <p class="mt-4 max-w-2xl mx-auto text-lg text-gray-600 dark:text-gray-300">
                The latest stories and announcements from the Genowa Delivery Unit.
            </p>
        </div>
        <?php if (!empty($latest_news)): ?>
            <div class="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-8">
                <?php foreach ($latest_news as $article): ?>
                    <a href="article.php?slug=<?php echo e($article['slug']); ?>" class="fade-in-section group block bg-white dark:bg-dark-card rounded-lg shadow-lg overflow-hidden transform hover:-translate-y-2 transition-all duration-300">
                        <div class="relative"><img src="admin/uploads/<?php echo e($article['featured_image'] ?? 'assets/images/placeholder.jpg'); ?>" alt="Featured image for <?php echo e($article['title']); ?>" class="h-56 w-full object-cover" onerror="this.onerror=null;this.src='assets/images/placeholder.jpg';"></div>
                        <div class="p-6"><p class="text-sm text-gray-500 dark:text-gray-400"><?php echo date('F j, Y', strtotime($article['created_at'])); ?></p><h3 class="mt-2 text-xl font-bold text-brand-blue dark:text-white group-hover:text-brand-purple transition-colors duration-300"><?php echo e($article['title']); ?></h3><div class="mt-2 text-sm text-gray-600 dark:text-gray-400 line-clamp-3"><?php echo strip_tags($article['content']); ?></div></div>
                    </a>
                <?php endforeach; ?>
            </div>
             <div class="mt-12 text-center">
                <a href="news.php" class="bg-brand-orange text-white font-bold py-3 px-6 rounded-full text-base hover:bg-opacity-90">View All News</a>
            </div>
        <?php else: ?>
             <div class="text-center bg-white dark:bg-dark-card p-12 rounded-lg shadow-md"><p class="text-gray-500 dark:text-gray-400">No news articles have been published yet. Please check back soon.</p></div>
        <?php endif; ?>
    </div>
</div>

<section class="bg-brand-purple text-white">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 py-16 text-center">
        <h2 class="text-3xl md:text-4xl font-extrabold">Become a Citizen Journalist</h2>
        <p class="mt-4 max-w-2xl mx-auto text-lg text-purple-200">
            Witnessed a project making an impact? Have a story to share about development in your ward? We invite you to submit your article for our platform.
        </p>
        <div class="mt-10">
            <a href="public_submit_article.php" class="bg-brand-orange text-white font-bold py-4 px-8 rounded-full text-lg hover:bg-opacity-90 transition duration-300 transform hover:scale-105 inline-block">
                Submit Your Story
            </a>
        </div>
    </div>
</section>

<section id="about" class="bg-brand-light-gray dark:bg-dark-bg py-16">
    <div class="container mx-auto px-4 sm:px-6 lg:px-8 text-center fade-in-section"><h2 class="text-3xl md:text-4xl font-extrabold text-brand-blue dark:text-white">About the Genowa Delivery Unit</h2><p class="mt-4 max-w-3xl mx-auto text-lg text-gray-600 dark:text-gray-300 leading-relaxed">The Genowa Delivery Unit is an initiative by the Homa Bay County Government, under the visionary leadership of H.E. Governor Gladys Wanga, to ensure the efficient, effective, and transparent implementation of development projects. Our mission is to translate the promises of the Genowa Manifesto into tangible results that improve the quality of life for every citizen of Homa Bay, truly showcasing our county's endless potential.</p></div>
</section>

<script>
document.addEventListener('DOMContentLoaded', () => {
    const animatedElements = document.querySelectorAll('.fade-in-section');
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                entry.target.classList.add('is-visible');
            }
        });
    }, { threshold: 0.1 });
    animatedElements.forEach(el => observer.observe(el));

    const countupElements = document.querySelectorAll('[data-countup]');
    countupElements.forEach(el => {
        const endValue = parseFloat(el.getAttribute('data-countup'));
        if(isNaN(endValue)) return;
        const duration = 2000;
        let startTimestamp = null;
        const step = timestamp => {
            if (!startTimestamp) startTimestamp = timestamp;
            const progress = Math.min((timestamp - startTimestamp) / duration, 1);
            let currentValue = progress * endValue;
            if (endValue % 1 !== 0) {
                 el.textContent = currentValue.toFixed(1);
            } else {
                 el.textContent = Math.floor(currentValue).toLocaleString();
            }
            if (progress < 1) {
                window.requestAnimationFrame(step);
            } else {
                // Use toLocaleString for the final value as well for consistency
                 el.textContent = endValue.toLocaleString('en-US', {minimumFractionDigits: (endValue % 1 !== 0) ? 1 : 0, maximumFractionDigits: (endValue % 1 !== 0) ? 1 : 0 });
            }
        };
        // Use an observer to start the animation only when the element is visible
        new IntersectionObserver((entries, obs) => {
            entries.forEach(entry => {
                if(entry.isIntersecting) {
                    window.requestAnimationFrame(step);
                    obs.unobserve(el); // Animate only once
                }
            });
        }, { threshold: 0.5 }).observe(el);
    });
});
</script>

<?php
include 'includes/footer.php';
?>