<?php
// FILE: admin/reports.php
// PURPOSE: To generate PDF reports for projects in a selected ward.

require_once 'functions.php';
require_once 'db.php';
require_once 'lib/fpdf/fpdf.php'; // Ensure FPDF library is here

// Security: Only SuperAdmins can access this page
require_login();
require_super_admin();

// --- Fetch data for form dropdowns ---
$wards = [];
$constituencies = [];
$pillars = [];
try {
    $wards = $pdo->query("SELECT id, name, constituency_id FROM wards ORDER BY name ASC")->fetchAll();
    $constituencies = $pdo->query("SELECT id, name FROM constituencies ORDER BY name ASC")->fetchAll();
    $pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY id ASC")->fetchAll();
} catch (PDOException $e) {
    $db_error = "Could not fetch location or pillar list: " . $e->getMessage();
}

// --- Check for form submission and generate PDF ---
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['generate_pdf'])) {
    
    // --- Get form data ---
    $report_level = $_POST['report_level'] ?? 'ward';
    $location_id = $_POST['location_id'] ?? null;
    $pillar_id = $_POST['pillar_id'] ?? null;
    $start_date = $_POST['start_date'] ?? null;
    $end_date = $_POST['end_date'] ?? null;

    if (empty($location_id) && !in_array($report_level, ['county', 'all'])) {
        $form_error = "Please select a location for this report level.";
    } else {
        try {
            // --- Build Dynamic SQL Query ---
            $sql = "SELECT p.*, w.name as ward_name, c.name as constituency_name
                    FROM projects p 
                    JOIN wards w ON p.ward_id = w.id
                    JOIN constituencies c ON w.constituency_id = c.id";
            
            $where_clauses = [];
            $params = [];
            $report_title = "County-Wide";
            $pillar_title_part = "";

            if ($report_level == 'constituency' && !empty($location_id)) {
                $where_clauses[] = "c.id = :location_id";
                $params['location_id'] = $location_id;
                $const_name_stmt = $pdo->prepare("SELECT name FROM constituencies WHERE id = :id");
                $const_name_stmt->execute(['id' => $location_id]);
                $report_title = $const_name_stmt->fetchColumn() . " Constituency";
            } elseif ($report_level == 'ward' && !empty($location_id)) {
                $where_clauses[] = "w.id = :location_id";
                $params['location_id'] = $location_id;
                $ward_name_stmt = $pdo->prepare("SELECT name FROM wards WHERE id = :id");
                $ward_name_stmt->execute(['id' => $location_id]);
                $report_title = $ward_name_stmt->fetchColumn() . " Ward";
            }

            if (!empty($pillar_id)) {
                $where_clauses[] = "p.manifesto_pillar_id = :pillar_id";
                $params['pillar_id'] = $pillar_id;
                $pillar_name_stmt = $pdo->prepare("SELECT title FROM manifesto_pillars WHERE id = :id");
                $pillar_name_stmt->execute(['id' => $pillar_id]);
                $pillar_title_part = " in the " . $pillar_name_stmt->fetchColumn() . " Sector";
            }

            if (!empty($start_date)) {
                $where_clauses[] = "p.start_date >= :start_date";
                $params['start_date'] = $start_date;
            }
            if (!empty($end_date)) {
                $where_clauses[] = "p.start_date <= :end_date";
                $params['end_date'] = $end_date;
            }

            if (!empty($where_clauses)) {
                $sql .= " WHERE " . implode(" AND ", $where_clauses);
            }
            $sql .= " ORDER BY c.name, w.name, p.name";

            $stmt = $pdo->prepare($sql);
            $stmt->execute($params);
            $projects = $stmt->fetchAll();

            if (empty($projects)) {
                 $form_error = "No projects found for the selected criteria.";
            } else {
                
                // --- PDF Generation using FPDF ---
                class PDF extends FPDF
                {
                    private $generator_name = 'N/A';

                    function setGenerator($name) {
                        $this->generator_name = $name;
                    }
                    
                    function Header()
                    {
                        // Standard Logo
                        $this->Image('assets/hbc_logo.png', 10, 8, 20);
                        $this->SetLineWidth(0.2);
                        $this->SetDrawColor(150, 150, 150);
                        $this->Line(35, 10, 35, 28);
                        $this->SetXY(38, 12);
                        $this->SetFont('Arial', 'B', 18);
                        $this->SetTextColor(29, 58, 138);
                        $this->Cell(40, 8, 'GENOWA', 0, 1, 'L');
                        $this->SetXY(38, 19);
                        $this->SetFont('Arial', 'B', 9);
                        $this->SetTextColor(249, 115, 22);
                        $this->Cell(40, 8, 'DELIVERY UNIT', 0, 1, 'L');
                        $this->SetXY(100, 12);
                        $this->SetFont('Arial', 'B', 14);
                        $this->SetTextColor(0,0,0);
                        $this->Cell(100, 10, 'Project Delivery Report', 0, 1, 'R');
                        $this->Ln(15);
                    }

                    function Footer()
                    {
                        $this->SetY(-15);
                        $this->SetFont('Arial', 'I', 8);
                        $this->SetTextColor(100, 100, 100);
                        $this->Cell(0, 5, "Report generated by: " . $this->generator_name, 0, 1, 'L');
                        $this->SetY(-15);
                        $this->Cell(0, 5, 'Page ' . $this->PageNo() . ' of {nb}', 0, 0, 'C');
                        $this->SetY(-15);
                        $this->Cell(0, 5, "Genowa Delivery Unit | Showcasing Homa Bay's Endless Potential", 0, 0, 'R');
                    }
                    
                    function LevelSummary($stats, $level_name)
                    {
                        $this->SetFont('Arial', 'B', 10);
                        $this->SetFillColor(243, 244, 246);
                        $this->SetTextColor(0,0,0);
                        // --- CHANGED THIS LINE ---
                        // Replaced Cell() with MultiCell() to allow the title to wrap if it's too long. Adjusted line height to 6.
                        $this->MultiCell(0, 6, 'Overview for: ' . $level_name, 'LTR', 'C', true);
                        $this->SetFont('Arial','',10);
                        $this->Cell(47.5, 8, 'Total Projects: ' . $stats['total'], 'L', 0, 'C', true);
                        $this->Cell(47.5, 8, 'Ongoing: ' . $stats['ongoing'], 0, 0, 'C', true);
                        $this->Cell(47.5, 8, 'Completed: ' . $stats['completed'], 0, 0, 'C', true);
                        $this->Cell(47.5, 8, 'Investment: KES ' . number_format($stats['investment'] / 1000000, 1) . 'M', 'R', 1, 'C', true);
                        $this->Cell(0, 0.5, '', 'LRB', 1, 'C', false);
                        $this->Ln(10);
                    }

                    function ProjectDetailTable($project)
                    {
                        $this->SetFont('Arial', 'B', 12);
                        $this->SetFillColor(29, 58, 138); // Brand Blue
                        $this->SetTextColor(255, 255, 255);
                        $this->Cell(0, 10, e($project['name']), 1, 1, 'C', true);
                        $this->SetTextColor(0, 0, 0);
                        $write_row = function($label, $value) {
                            $this->SetFont('Arial', 'B', 10);
                            $this->SetFillColor(243, 244, 246);
                            $this->Cell(45, 8, $label, 'TL', 0, 'L', true);
                            $this->SetFont('Arial', '', 10);
                            $this->MultiCell(0, 8, e($value ?: 'N/A'), 'TR');
                        };
                        $write_row('Status:', $project['status']);
                        $write_row('Location:', e($project['ward_name']) . ", " . e($project['constituency_name']));
                        $write_row('Scope of Work:', $project['description']);
                        $write_row('Beneficiaries:', $project['beneficiaries_summary']);
                        $write_row('Start Date:', $project['start_date'] ? date('M j, Y', strtotime($project['start_date'])) : 'N/A');
                        $write_row('Expected Completion:', $project['expected_completion_date'] ? date('M j, Y', strtotime($project['expected_completion_date'])) : 'N/A');
                        $write_row('Actual Completion:', $project['actual_completion_date'] ? date('M j, Y', strtotime($project['actual_completion_date'])) : 'N/A');
                        $write_row('Budget Allocated:', 'KES ' . number_format($project['budget_allocated'], 2));
                        $write_row('Budget Spent:', 'KES ' . number_format($project['budget_spent'], 2));
                        $this->Cell(0, 0.1, '', 'T', 1);
                    }
                }

                $report_stats = [
                    'total' => count($projects),
                    'ongoing' => 0,
                    'completed' => 0,
                    'investment' => 0,
                ];
                foreach ($projects as $project) {
                    if ($project['status'] == 'Ongoing') $report_stats['ongoing']++;
                    if ($project['status'] == 'Completed') $report_stats['completed']++;
                    $report_stats['investment'] += $project['budget_allocated'];
                }

                $pdf = new PDF('P', 'mm', 'A4');
                $pdf->setGenerator($_SESSION['name']);
                $pdf->AliasNbPages();
                $pdf->AddPage();
                
                $full_report_title = 'Project Report for: ' . $report_title . $pillar_title_part;
                $pdf->SetFont('Arial', 'B', 14);
                // --- CHANGED THIS LINE ---
                // Replaced Cell() with MultiCell() to allow the title to wrap. Adjusted line height to 8.
                $pdf->MultiCell(0, 8, $full_report_title, 0, 'C');
                $pdf->SetFont('Arial', '', 10);
                $pdf->Cell(0, 5, 'Report Generated on: ' . date('F j, Y, g:i a'), 0, 1, 'C');
                if ($start_date || $end_date) {
                    $pdf->Cell(0, 5, 'Period: ' . ($start_date ?: 'Start') . ' to ' . ($end_date ?: 'End'), 0, 1, 'C');
                }
                $pdf->Ln(5);

                $pdf->LevelSummary($report_stats, $report_title . $pillar_title_part);
                
                foreach ($projects as $project) {
                    if ($pdf->GetY() > 220) { $pdf->AddPage(); }
                    $pdf->ProjectDetailTable($project);
                    $pdf->Ln(5);
                }
                
                // NEW: Log the action before sending the PDF
                log_action($pdo, 'Generated PDF Report', 'Report', intval($location_id));

                $pdf->Output('I', 'Report.pdf');
                exit;
            }
        } catch (PDOException $e) {
            $db_error = "Error generating report: " . $e->getMessage();
        }
    }
}

include 'includes/header.php';
?>
<div class="min-h-full">
    <header class="bg-white shadow">
        <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8">
            <h1 class="text-3xl font-bold tracking-tight text-gray-900">Generate PDF Reports</h1>
        </div>
    </header>

    <main>
        <div class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
            <div class="max-w-xl mx-auto">
                 <div class="bg-white rounded-lg shadow-lg p-8">
                    <form action="reports.php" method="post" target="_blank" class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="report_level" class="block text-sm font-medium text-gray-700">Report Level</label>
                                <select name="report_level" id="report_level" class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3">
                                    <option value="ward">Ward</option>
                                    <option value="constituency">Constituency</option>
                                    <option value="county">County-Wide</option>
                                </select>
                            </div>

                            <div>
                                <label for="location_id" class="block text-sm font-medium text-gray-700">Location</label>
                                <select name="location_id" id="location_id" class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3">
                                </select>
                            </div>
                        </div>

                        <div>
                            <label for="pillar_id" class="block text-sm font-medium text-gray-700">Sector (Manifesto Pillar)</label>
                            <select name="pillar_id" id="pillar_id" class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3">
                                <option value="">All Sectors</option>
                                <?php foreach ($pillars as $pillar): ?>
                                    <option value="<?php echo e($pillar['id']); ?>"><?php echo e($pillar['title']); ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <label for="start_date" class="block text-sm font-medium text-gray-700">Start Date (Optional)</label>
                                <input type="date" name="start_date" id="start_date" class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3">
                            </div>
                             <div>
                                <label for="end_date" class="block text-sm font-medium text-gray-700">End Date (Optional)</label>
                                <input type="date" name="end_date" id="end_date" class="mt-1 block w-full rounded-lg border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 px-4 py-3">
                            </div>
                        </div>

                        <div class="pt-2">
                            <button type="submit" name="generate_pdf" class="w-full inline-flex justify-center items-center rounded-md border border-transparent bg-brand-orange px-6 py-3 text-base font-medium text-white shadow-sm hover:bg-opacity-90">
                                Preview &amp; Generate PDF Report
                            </button>
                        </div>
                    </form>
                 </div>
            </div>
        </div>
    </main>
</div>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const reportLevelSelect = document.getElementById('report_level');
    const locationSelect = document.getElementById('location_id');
    const wards = <?php echo json_encode($wards); ?>;
    const constituencies = <?php echo json_encode($constituencies); ?>;

    function updateLocationOptions() {
        const level = reportLevelSelect.value;
        locationSelect.innerHTML = '';
        if (level === 'county') {
            locationSelect.disabled = true;
            locationSelect.classList.add('bg-gray-100');
            const option = document.createElement('option');
            option.value = 'all';
            option.textContent = 'All of Homa Bay County';
            locationSelect.appendChild(option);
        } else {
            locationSelect.disabled = false;
            locationSelect.classList.remove('bg-gray-100');
            const placeholder = document.createElement('option');
            placeholder.value = '';
            placeholder.textContent = `-- Select a ${level.charAt(0).toUpperCase() + level.slice(1)} --`;
            locationSelect.appendChild(placeholder);
            const options = level === 'ward' ? wards : constituencies;
            options.forEach(item => {
                const option = document.createElement('option');
                option.value = item.id;
                option.textContent = item.name;
                locationSelect.appendChild(option);
            });
        }
    }
    reportLevelSelect.addEventListener('change', updateLocationOptions);
    updateLocationOptions();
});
</script>
<?php include 'includes/footer.php'; ?>