<?php
// FILE: admin/news_dashboard.php
// PURPOSE: To view, manage, and publish news articles.

require_once 'functions.php';
require_once 'db.php';

// Security: Only Admins and SuperAdmins can access
require_login();
if (!is_admin() && !is_super_admin()) {
    header("location: dashboard.php");
    exit;
}

// --- Setup Pagination ---
$page = isset($_GET['page']) ? (int)$_GET['page'] : 1;
$limit = 15;
$offset = ($page - 1) * $limit;

$articles = [];
$total_articles = 0;

try {
    // Get total count for pagination
    $count_stmt = $pdo->query("SELECT COUNT(id) FROM news_articles");
    $total_articles = $count_stmt->fetchColumn();
    $total_pages = ceil($total_articles / $limit);

    // Fetch articles for the current page
    $sql = "SELECT a.*, u.name as author_name 
            FROM news_articles a
            JOIN users u ON a.author_id = u.id
            ORDER BY a.updated_at DESC 
            LIMIT :limit OFFSET :offset";
    $stmt = $pdo->prepare($sql);
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $articles = $stmt->fetchAll();

} catch (PDOException $e) {
    $db_error = "Error fetching articles: " . $e->getMessage();
}

include 'includes/header.php';
?>

<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8 flex justify-between items-center">
        <div>
            <h1 class="text-lg font-semibold leading-6 text-gray-900">News & Updates</h1>
            <p class="mt-1 text-sm text-gray-500">Total articles: <?php echo e(number_format($total_articles)); ?></p>
        </div>
        <a href="manage_article.php" class="inline-flex items-center gap-x-2 rounded-md bg-brand-orange px-3.5 py-2.5 text-sm font-semibold text-white shadow-sm hover:bg-orange-500">
            Write New Article
        </a>
    </div>
</header>
    
<div class="overflow-x-auto bg-white rounded-lg shadow mt-6">
    <?php if (isset($db_error)): ?>
        <div class="p-4 bg-red-100 text-red-700" role="alert"><p><?php echo e($db_error); ?></p></div>
    <?php elseif (empty($articles)): ?>
        <div class="text-center p-12">
            <h3 class="mt-2 text-lg font-medium text-gray-900">No articles found.</h3>
            <p class="mt-1 text-sm text-gray-500">Get started by writing a new article.</p>
        </div>
    <?php else: ?>
        <table class="min-w-full divide-y divide-gray-200">
            <thead class="bg-gray-50">
                <tr>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Title</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Author</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Status</th>
                    <th scope="col" class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">Last Updated</th>
                    <th scope="col" class="relative px-6 py-3"><span class="sr-only">Actions</span></th>
                </tr>
            </thead>
            <tbody class="bg-white divide-y divide-gray-200">
                <?php foreach ($articles as $article): ?>
                    <tr>
                        <td class="px-6 py-4 whitespace-nowrap text-sm font-medium text-gray-900"><?php echo e($article['title']); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo e($article['author_name']); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap">
                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                <?php echo ($article['status'] == 'Published') ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'; ?>">
                                <?php echo e($article['status']); ?>
                            </span>
                        </td>
                        <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo date('M j, Y', strtotime($article['updated_at'])); ?></td>
                        <td class="px-6 py-4 whitespace-nowrap text-right text-sm font-medium">
                            <a href="manage_article.php?id=<?php echo $article['id']; ?>" class="text-brand-purple hover:text-brand-orange">Edit</a>
                        </td>
                    </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    <?php endif; ?>
</div>

<!-- Pagination -->
<div class="mt-6 flex justify-between items-center">
    <?php if ($page > 1): ?>
        <a href="?page=<?php echo $page - 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Previous</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Previous</span>
    <?php endif; ?>
    <div class="text-sm text-gray-700">Page <?php echo $page; ?> of <?php echo $total_pages; ?></div>
    <?php if ($page < $total_pages): ?>
        <a href="?page=<?php echo $page + 1; ?>" class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50">Next</a>
    <?php else: ?>
        <span class="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-400 bg-gray-100 cursor-not-allowed">Next</span>
    <?php endif; ?>
</div>

<?php include 'includes/footer.php'; ?>
