<?php
// FILE: admin/manage_project.php
require_once 'functions.php';
require_once 'db.php';
require_login();

$project_id = $_GET['id'] ?? null;
$project = null;
$errors = [];
$success_message = '';
$page_title = "Add New Project";
$is_edit_mode = ($project_id !== null);

$can_edit = false;
$user_can_approve = is_super_admin() || is_admin() || is_approver();

if ($is_edit_mode) {
    $stmt = $pdo->prepare("SELECT * FROM projects WHERE id = :id");
    $stmt->execute(['id' => $project_id]);
    $project = $stmt->fetch();
    if (!$project) {
        header("location: dashboard.php");
        exit;
    }
    if (is_inputer() && ($project['created_by'] != $_SESSION['id'] || $project['is_approved'] == 1)) {
        header("location: dashboard.php");
        exit;
    }
    if (is_super_admin() || is_admin()) $can_edit = true;
    elseif (is_inputer()) $can_edit = ($project['is_approved'] == 0);

    $page_title = ($project['is_approved'] == 0 && $user_can_approve && !$can_edit) ? "Review Project" : "Manage Project";
    $page_title .= ": " . e($project['name']);
} else {
    $can_edit = is_super_admin() || is_admin() || is_inputer();
    if (!$can_edit) {
        header("location: dashboard.php");
        exit;
    }
}

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    // --- Action: Approve Project ---
    if (isset($_POST['approve_project']) && $user_can_approve && $is_edit_mode) {
        try {
            $sql = "UPDATE projects SET is_approved = 1, approved_by = :approved_by, approved_at = NOW() WHERE id = :id";
            $stmt = $pdo->prepare($sql);
            $stmt->execute(['approved_by' => $_SESSION['id'], 'id' => $project_id]);
            log_action($pdo, 'Approved Project', 'Project', $project_id);
            header("location: dashboard.php?success=approved");
            exit;
        } catch (PDOException $e) {
            $errors['db'] = "Database Error: " . $e->getMessage();
        }
    }
    // --- Action: Save Project Details ---
    if (isset($_POST['save_details']) && $can_edit) {
        $current_project_id = $_POST['id'] ?? null;
        $is_edit_mode_on_post = !empty($current_project_id);

        $project_data = [
            'name' => trim($_POST['name']), 'description' => trim($_POST['description']),
            'status' => trim($_POST['status']), 'budget_allocated' => trim($_POST['budget_allocated']) ?: 0,
            'budget_spent' => empty(trim($_POST['budget_spent'])) ? null : trim($_POST['budget_spent']),
            'start_date' => empty(trim($_POST['start_date'])) ? null : trim($_POST['start_date']),
            'expected_completion_date' => empty(trim($_POST['expected_completion_date'])) ? null : trim($_POST['expected_completion_date']),
            'actual_completion_date' => empty(trim($_POST['actual_completion_date'])) ? null : trim($_POST['actual_completion_date']),
            'beneficiaries_summary' => trim($_POST['beneficiaries_summary']), 'contractor_name' => trim($_POST['contractor_name']),
            'manifesto_pillar_id' => trim($_POST['manifesto_pillar_id']),
            'ward_id' => is_super_admin() ? trim($_POST['ward_id']) : $_SESSION['ward_id'],
            'latitude' => trim($_POST['latitude']) ?: null, 'longitude' => trim($_POST['longitude']) ?: null,
        ];

        if (empty($project_data['name'])) $errors['name'] = "Project name is required.";
        if (empty($errors)) {
            try {
                if ($is_edit_mode_on_post) {
                    $sql = "UPDATE projects SET name = :name, description = :description, status = :status, budget_allocated = :budget_allocated, budget_spent = :budget_spent, start_date = :start_date, expected_completion_date = :expected_completion_date, actual_completion_date = :actual_completion_date, beneficiaries_summary = :beneficiaries_summary, contractor_name = :contractor_name, ward_id = :ward_id, latitude = :latitude, longitude = :longitude, manifesto_pillar_id = :manifesto_pillar_id, last_updated_by = :last_updated_by WHERE id = :id";
                    $project_data['last_updated_by'] = $_SESSION['id'];
                    $project_data['id'] = $current_project_id;
                } else {
                    $sql = "INSERT INTO projects (name, description, status, budget_allocated, budget_spent, start_date, expected_completion_date, actual_completion_date, beneficiaries_summary, contractor_name, ward_id, latitude, longitude, manifesto_pillar_id, created_by, last_updated_by) VALUES (:name, :description, :status, :budget_allocated, :budget_spent, :start_date, :expected_completion_date, :actual_completion_date, :beneficiaries_summary, :contractor_name, :ward_id, :latitude, :longitude, :manifesto_pillar_id, :created_by, :last_updated_by)";
                    $project_data['created_by'] = $_SESSION['id'];
                    $project_data['last_updated_by'] = $_SESSION['id'];
                }
                $stmt = $pdo->prepare($sql);
                $stmt->execute($project_data);
                $new_id = $is_edit_mode_on_post ? $current_project_id : $pdo->lastInsertId();
                log_action($pdo, $is_edit_mode_on_post ? 'Updated Project' : 'Created Project', 'Project', $new_id);

                // Force user to go to Manage Photos after details are saved
                header("location: manage_project.php?id=" . $new_id . "&tab=photos&success=saved");
                exit;
            } catch (PDOException $e) {
                $errors['db'] = "Database Error: " . $e->getMessage();
            }
        }
    }
}

// --- Fetch data for dropdowns and photos---
$wards = [];
$manifesto_pillars = [];
$project_photos = [];
try {
    $wards = $pdo->query("SELECT id, name FROM wards ORDER BY name ASC")->fetchAll();
    $manifesto_pillars = $pdo->query("SELECT id, title FROM manifesto_pillars ORDER BY id ASC")->fetchAll();
    if ($is_edit_mode) {
        $photos_stmt = $pdo->prepare("SELECT * FROM project_photos WHERE project_id = :id ORDER BY uploaded_at DESC");
        $photos_stmt->execute(['id' => $project_id]);
        $project_photos = $photos_stmt->fetchAll();
    }
} catch (PDOException $e) {
    $errors['db_load'] = "Could not fetch page data.";
}

include 'includes/header.php';
?>
<header class="bg-white shadow-sm">
    <div class="mx-auto max-w-7xl px-4 py-4 sm:px-6 lg:px-8">
        <h1 class="text-lg font-semibold leading-6 text-gray-900 truncate"><?php echo e($page_title); ?></h1>
        <?php if ($project && $project['is_approved'] == 0): ?>
            <span class="text-xs font-bold text-yellow-800 bg-yellow-100 py-1 px-2 rounded-full">PENDING APPROVAL</span>
        <?php elseif($project): ?>
             <span class="text-xs font-bold text-green-800 bg-green-100 py-1 px-2 rounded-full">APPROVED</span>
        <?php endif; ?>
    </div>
</header>

<div class="border-b border-gray-200">
    <nav class="-mb-px flex space-x-8" aria-label="Tabs">
        <button id="tab-details" class="tab-button whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm border-brand-purple text-brand-purple">Project Details</button>
        <button id="tab-photos" class="tab-button whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 <?php if (!$is_edit_mode) echo 'cursor-not-allowed opacity-60'; ?>" <?php if (!$is_edit_mode) echo 'disabled'; ?>>Manage Photos</button>
        <button id="tab-danger" class="tab-button whitespace-nowrap py-4 px-1 border-b-2 font-medium text-sm border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 <?php if (!$is_edit_mode) echo 'cursor-not-allowed opacity-60'; ?>" <?php if (!$is_edit_mode) echo 'disabled'; ?>>Danger Zone</button>
    </nav>
</div>

<div class="mt-6">
    <!-- Project Details Tab -->
    <div id="content-details" class="tab-content">
        <?php include 'partials/project_details_form.php'; ?>
    </div>
    <!-- Manage Photos Tab -->
    <div id="content-photos" class="tab-content hidden">
        <?php include 'partials/project_photos_form.php'; ?>
    </div>
    <!-- Danger Zone Tab -->
    <div id="content-danger" class="tab-content hidden">
       <div class="bg-white rounded-lg shadow-lg p-8 border border-red-200">
           <h3 class="text-lg font-medium leading-6 text-red-700">Delete Project</h3>
           <div class="mt-2 max-w-xl text-sm text-gray-500">
               <p>Once you delete a project, all of its data, including all associated photos, will be permanently removed. This action cannot be undone.</p>
           </div>
           <div class="mt-5">
               <a href="project_delete.php?id=<?php echo e($project['id'] ?? ''); ?>" class="inline-flex items-center justify-center px-4 py-2 border border-transparent font-medium rounded-md text-white bg-red-600 hover:bg-red-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-red-500 sm:text-sm" onclick="return confirm('Are you absolutely sure you want to delete this project? This will delete all its data and photos permanently.');">
                   Delete This Project
               </a>
           </div>
       </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const tabs = document.querySelectorAll('.tab-button');
    const contents = document.querySelectorAll('.tab-content');

    function switchTab(tabId) {
        tabs.forEach(item => {
            item.classList.remove('border-brand-purple', 'text-brand-purple');
            item.classList.add('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
        });
        const activeTab = document.getElementById(tabId);
        if (activeTab) {
            activeTab.classList.add('border-brand-purple', 'text-brand-purple');
            activeTab.classList.remove('border-transparent', 'text-gray-500', 'hover:text-gray-700', 'hover:border-gray-300');
        }
        contents.forEach(content => {
            content.classList.add('hidden');
        });
        const targetContentId = tabId.replace('tab-', 'content-');
        const activeContent = document.getElementById(targetContentId);
        if (activeContent) {
            activeContent.classList.remove('hidden');
            window.scrollTo({top: 0, behavior: 'smooth'});
        }
    }

    tabs.forEach(tab => {
        tab.addEventListener('click', () => {
            if (tab.disabled) return;
            history.replaceState(null, null, '#' + tab.id);
            switchTab(tab.id);
        });
    });

    // Default to tab from URL hash or ?tab= param
    let startTab = 'tab-details';
    const urlTab = new URLSearchParams(window.location.search).get('tab');
    if (urlTab && document.getElementById('tab-' + urlTab)) {
        startTab = 'tab-' + urlTab;
    } else if (window.location.hash && document.getElementById(window.location.hash.substring(1))) {
        startTab = window.location.hash.substring(1);
    }
    switchTab(startTab);
});
</script>

<?php include 'includes/footer.php'; ?>
