<?php
// FILE: admin/generate_infographic.php
// PURPOSE: Dynamically generates a shareable infographic for a selected project.

require_once 'functions.php';
require_once 'db.php';

require_login();

// --- Get Project ID from URL ---
$project_id = $_GET['id'] ?? null;
$project = null;
$all_photos = [];
$completion_percentage = 0;
$all_projects_list = [];

try {
    // Fetch list of all approved projects for the dropdown
    $projects_list_stmt = $pdo->query("SELECT id, name FROM projects WHERE is_approved = 1 ORDER BY name ASC");
    $all_projects_list = $projects_list_stmt->fetchAll();

    // If an ID is provided, fetch the specific project data
    if ($project_id) {
        $sql = "SELECT p.*, w.name as ward_name, c.name as constituency_name
                FROM projects p
                JOIN wards w ON p.ward_id = w.id
                JOIN constituencies c ON w.constituency_id = c.id
                WHERE p.id = :id AND p.is_approved = 1";
        $stmt = $pdo->prepare($sql);
        $stmt->execute(['id' => $project_id]);
        $project = $stmt->fetch();

        if ($project) {
            $photos_stmt = $pdo->prepare("SELECT id, image_url, photo_type FROM project_photos WHERE project_id = :project_id ORDER BY photo_type, uploaded_at DESC");
            $photos_stmt->execute(['project_id' => $project_id]);
            $all_photos = $photos_stmt->fetchAll();

            if ($project['status'] === 'Completed') {
                $completion_percentage = 100;
            } elseif ($project['status'] === 'Ongoing' && $project['start_date'] && $project['expected_completion_date']) {
                $today = new DateTime();
                $start_date = new DateTime($project['start_date']);
                $expected_date = new DateTime($project['expected_completion_date']);
                if ($start_date <= $expected_date && $today >= $start_date) {
                    $total_duration = $start_date->diff($expected_date)->days;
                    $elapsed_duration = $start_date->diff($today)->days;
                    $completion_percentage = ($total_duration > 0) ? min(100, floor(($elapsed_duration / $total_duration) * 100)) : 100;
                }
            }
        }
    }
} catch (PDOException $e) {
    $db_error = "Database error: " . $e->getMessage();
}

include 'includes/header.php';
?>
<!-- Include libraries -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js"></script>
<link href="https://cdn.jsdelivr.net/npm/tom-select@2.2.2/dist/css/tom-select.bootstrap5.css" rel="stylesheet">
<script src="https://cdn.jsdelivr.net/npm/tom-select@2.2.2/dist/js/tom-select.complete.min.js"></script>
<style>
    /* Custom styles for TomSelect */
    .ts-control { padding: 0.75rem 1rem !important; border-radius: 0.5rem !important; }
     .layout-banner-bottom .layout-content-overlay, .layout-banner-bottom .layout-content-orange-theme { display: none; }
    .layout-classic-overlay .layout-content-banner, .layout-classic-overlay .layout-content-orange-theme { display: none; }
    .layout-orange-theme .layout-content-banner, .layout-orange-theme .layout-content-overlay { display: none; }
    #infographic-container {
        font-family: 'Inter', sans-serif;
    }
    .text-shadow {
        text-shadow: 2px 2px 4px rgba(0,0,0,0.6);
    }
</style>


<header class="bg-white shadow">
    <div class="mx-auto max-w-7xl px-4 py-6 sm:px-6 lg:px-8">
        <h1 class="text-3xl font-bold tracking-tight text-gray-900">Generate Infographic</h1>
    </div>
</header>
<main class="mx-auto max-w-7xl py-6 sm:px-6 lg:px-8">
    
    <!-- Project Selection Form -->
    <div class="bg-white p-6 rounded-lg shadow-lg mb-8">
        <form action="generate_infographic.php" method="get">
            <label for="project-select" class="block text-lg font-medium text-gray-900 mb-2">Select a Project to Generate an Infographic</label>
            <div class="flex items-center space-x-2">
                <select id="project-select" name="id" placeholder="Search for a project..." class="flex-grow">
                    <option value="">Search for a project...</option>
                    <?php foreach ($all_projects_list as $p): ?>
                        <option value="<?php echo e($p['id']); ?>" <?php if($project_id == $p['id']) echo 'selected'; ?>><?php echo e($p['name']); ?></option>
                    <?php endforeach; ?>
                </select>
                <button type="submit" class="rounded-md bg-brand-blue px-4 py-3 text-sm font-semibold text-white shadow-sm hover:bg-brand-purple">Load Project</button>
            </div>
        </form>
    </div>

    <?php if ($project): // Only show the generator if a project is loaded ?>
    <div class="grid grid-cols-1 lg:grid-cols-3 gap-8">
        <!-- Left: Infographic Preview -->
        <div class="lg:col-span-2">
             <h2 class="text-xl font-bold mb-4">Live Preview</h2>
                <div id="infographic-container" class="layout-banner-bottom w-full aspect-square bg-gray-800 text-white relative flex flex-col shadow-2xl overflow-hidden">
                    <div class="absolute inset-0 z-0">
                        <img id="bg-image" src="<?php echo !empty($all_photos) ? 'uploads/' . e($all_photos[0]['image_url']) : ''; ?>" class="w-full h-full object-cover filter brightness-105 contrast-110" crossorigin="anonymous" onerror="this.style.display='none'">
                        <div class="absolute inset-0 bg-gradient-to-t from-black/70 via-black/30 to-transparent"></div>
                    </div>
                    <div class="layout-content-banner flex-grow flex flex-col justify-between p-6">
                        <div class="relative z-10 flex justify-between items-start">
                            <div class="logo-container flex items-center space-x-4">
                                <div class="hbc-logo-wrapper"><img src="assets/hbc_logo.png" alt="Homa Bay County Logo" class="h-12 w-auto" crossorigin="anonymous"></div>
                                <div class="logo-divider w-px h-12 bg-white/50"></div>
                                <div class="genowa-logo-wrapper"><p class="font-black text-3xl text-white">GENOWA</p><p class="font-semibold text-lg text-brand-orange tracking-wider -mt-2">DELIVERY UNIT</p></div>
                            </div>
                            <div class="px-3 py-1 text-sm font-bold inline-block rounded-full bg-white/20 backdrop-blur-sm <?php if ($project['status'] === 'Ongoing') echo 'text-blue-300'; ?>"><?php echo e($project['status']); ?></div>
                        </div>
                        <div class="relative z-10">
                            <div class="bg-brand-blue/90 p-4 rounded-lg backdrop-blur-sm shadow-lg border border-white/10">
                                <p class="text-lg font-semibold text-brand-orange"><?php echo e($project['ward_name']); ?> Ward</p>
                                <h3 class="text-3xl font-black leading-tight text-white"><?php echo e($project['name']); ?></h3>
                                <?php if (($project['status'] === 'Ongoing' && $completion_percentage > 0) || $project['status'] === 'Completed'): ?>
                                    <div class="mt-3"><div class="w-full bg-white/20 rounded-full h-2"><div class="bg-brand-orange h-2 rounded-full" style="width: <?php echo $completion_percentage; ?>%"></div></div><p class="text-right text-xs font-bold mt-1 text-gray-200"><?php echo $completion_percentage; ?>% <?php echo ($project['status'] === 'Completed') ? 'Complete' : 'Timeline Progress'; ?></p></div>
                                <?php endif; ?>
                                <p class="infographic-caption mt-2 text-base text-gray-200 leading-snug"></p>
                                <p class="hashtag-container text-right text-xs font-bold mt-2 text-brand-orange/80">#GladysWangaDelivers</p>
                            </div>
                        </div>
                    </div>
                     <div class="layout-content-overlay flex-grow flex flex-col justify-between p-6">
                        <div class="relative z-10 flex justify-between items-start">
                             <div class="logo-container flex items-center space-x-4">
                                <div class="hbc-logo-wrapper"><img src="assets/hbc_logo.png" alt="Homa Bay County Logo" class="h-12 w-auto text-shadow" crossorigin="anonymous"></div>
                                <div class="logo-divider w-px h-12 bg-white/50"></div>
                                <div class="genowa-logo-wrapper"><p class="font-black text-3xl text-white text-shadow">GENOWA</p><p class="font-semibold text-lg text-brand-orange tracking-wider -mt-2 text-shadow">DELIVERY UNIT</p></div>
                            </div>
                            <div class="px-3 py-1 text-sm font-bold inline-block rounded-full bg-white/20 backdrop-blur-sm text-shadow <?php if ($project['status'] === 'Ongoing') echo 'text-blue-300'; ?>"><?php echo e($project['status']); ?></div>
                        </div>
                        <div class="relative z-10 text-shadow">
                            <p class="text-lg font-semibold text-brand-orange"><?php echo e($project['ward_name']); ?> Ward</p>
                            <h3 class="text-4xl font-black leading-tight text-white"><?php echo e($project['name']); ?></h3>
                            <?php if (($project['status'] === 'Ongoing' && $completion_percentage > 0) || $project['status'] === 'Completed'): ?>
                                <div class="mt-3"><div class="w-full bg-white/20 rounded-full h-2"><div class="bg-brand-orange h-2 rounded-full" style="width: <?php echo $completion_percentage; ?>%"></div></div><p class="text-right text-sm font-bold mt-1 text-white"><?php echo $completion_percentage; ?>% <?php echo ($project['status'] === 'Completed') ? 'Complete' : 'Timeline'; ?></p></div>
                            <?php endif; ?>
                            <p class="infographic-caption mt-2 text-base text-gray-200 leading-snug"></p>
                            <p class="hashtag-container text-right text-sm font-bold mt-2 text-brand-orange/80">#GladysWangaDelivers</p>
                        </div>
                    </div>
                    <div class="layout-content-orange-theme flex-grow flex flex-col justify-between">
                        <div class="p-6 relative z-10 flex justify-between items-start">
                             <div class="logo-container flex items-center space-x-4">
                                <div class="hbc-logo-wrapper"><img src="assets/hbc_logo.png" alt="Homa Bay County Logo" class="h-12 w-auto" crossorigin="anonymous"></div>
                                <div class="logo-divider w-px h-12 bg-white/50"></div>
                                <div class="genowa-logo-wrapper"><p class="font-black text-3xl text-white">GENOWA</p><p class="font-semibold text-lg text-white tracking-wider -mt-2">DELIVERY UNIT</p></div>
                            </div>
                            <div class="px-3 py-1 text-sm font-bold inline-block rounded-full bg-white/20 backdrop-blur-sm <?php if ($project['status'] === 'Ongoing') echo 'text-blue-300'; ?>"><?php echo e($project['status']); ?></div>
                        </div>
                         <div class="relative z-10 bg-brand-orange p-6 text-white">
                            <p class="text-lg font-bold text-white"><?php echo e($project['ward_name']); ?> Ward</p>
                            <h3 class="text-3xl font-black leading-tight text-brand-blue"><?php echo e($project['name']); ?></h3>
                            <p class="infographic-caption mt-2 text-base text-white leading-snug"></p>
                            <p class="hashtag-container text-right text-sm font-bold mt-2 text-white/80">#GladysWangaDelivers</p>
                        </div>
                    </div>
                </div>
        </div>

        <!-- Right: Controls -->
        <div>
            <h2 class="text-xl font-bold mb-4">Customize Infographic</h2>
            <div class="bg-white p-6 rounded-lg shadow space-y-6">
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">1. Select Background Image</label>
                    <div id="thumbnail-selector" class="grid grid-cols-4 gap-2 h-32 overflow-y-auto bg-gray-100 p-2 rounded-md">
                       <?php if(empty($all_photos)): ?>
                           <p class="col-span-full text-sm text-gray-500 text-center my-auto">No photos available for this project.</p>
                       <?php endif; ?>
                    </div>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">2. Choose a Layout</label>
                    <fieldset class="mt-2"><div class="space-y-4">
                            <div class="flex items-center"><input id="layout-banner-bottom" name="layout" type="radio" value="layout-banner-bottom" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple" checked><label for="layout-banner-bottom" class="ml-3 block text-sm font-medium text-gray-700">Banner Bottom</label></div>
                            <div class="flex items-center"><input id="layout-classic-overlay" name="layout" type="radio" value="layout-classic-overlay" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple"><label for="layout-classic-overlay" class="ml-3 block text-sm font-medium text-gray-700">Classic Overlay</label></div>
                            <div class="flex items-center"><input id="layout-orange-theme" name="layout" type="radio" value="layout-orange-theme" class="h-4 w-4 border-gray-300 text-brand-purple focus:ring-brand-purple"><label for="layout-orange-theme" class="ml-3 block text-sm font-medium text-gray-700">Orange Theme</label></div>
                    </div></fieldset>
                </div>
                 <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">3. Toggle Elements</label>
                    <div class="space-y-2">
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-hbc-logo" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-hbc-logo" class="font-medium text-gray-700">Show Homa Bay Logo</label></div></div>
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-logo" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-logo" class="font-medium text-gray-700">Show Genowa Logo</label></div></div>
                        <div class="relative flex items-start"><div class="flex h-6 items-center"><input id="toggle-hashtag" type="checkbox" class="h-4 w-4 rounded border-gray-300 text-brand-purple focus:ring-brand-purple" checked></div><div class="ml-3 text-sm"><label for="toggle-hashtag" class="font-medium text-gray-700">Show #GladysWangaDelivers</label></div></div>
                    </div>
                </div>
                <div>
                    <label for="caption-input" class="block text-sm font-medium text-gray-700 mb-2">4. Enter a Caption</label>
                    <textarea id="caption-input" rows="5" maxlength="500" class="w-full rounded-md border-gray-300 shadow-sm focus:border-brand-purple focus:ring-2 focus:ring-brand-purple focus:ring-opacity-50 transition-shadow duration-200" placeholder="Type text here to generate caption for the infographic..."></textarea>
                    <div id="char-counter" class="text-right text-sm text-gray-500 mt-1">0 / 500</div>
                </div>
                <div>
                    <label class="block text-sm font-medium text-gray-700 mb-2">5. Download Image</label>
                    <button id="download-btn" class="w-full inline-flex justify-center items-center rounded-md border border-transparent bg-brand-orange px-6 py-3 text-base font-medium text-white shadow-sm hover:bg-opacity-90 disabled:bg-gray-400" disabled>
                        Download as PNG
                    </button>
                </div>
            </div>
        </div>
    </div>
    <?php elseif ($project_id): ?>
        <div class="bg-red-100 text-red-700 p-4 rounded-lg">Project with ID <?php echo e($project_id); ?> not found or not approved.</div>
    <?php endif; ?>

</main>

<script>
document.addEventListener('DOMContentLoaded', function() {
    new TomSelect('#project-select',{
        create: false,
        sortField: {
            field: "text",
            direction: "asc"
        }
    });
    
    <?php if ($project): ?>
        const infographicContainer = document.getElementById('infographic-container');
        const thumbnailSelector = document.getElementById('thumbnail-selector');
        const layoutRadios = document.querySelectorAll('input[name="layout"]');
        const toggleLogoCheckbox = document.getElementById('toggle-logo');
        const toggleHbcLogoCheckbox = document.getElementById('toggle-hbc-logo');
        const toggleHashtagCheckbox = document.getElementById('toggle-hashtag');
        const captionInput = document.getElementById('caption-input');
        const downloadBtn = document.getElementById('download-btn');
        const charCounter = document.getElementById('char-counter');
        const maxChars = 500;
        const allPhotos = <?php echo json_encode($all_photos); ?>;

        allPhotos.forEach((photo, index) => {
            const thumb = document.createElement('img');
            thumb.src = `uploads/${photo.image_url}`;
            thumb.className = "w-full h-auto object-cover rounded-md cursor-pointer border-2 border-transparent hover:border-brand-purple";
            if (index === 0) {
                thumb.classList.add('border-brand-purple');
            }
            thumb.addEventListener('click', () => {
                const newBg = `linear-gradient(to top, rgba(0,0,0,0.7), rgba(0,0,0,0.3), transparent), url(uploads/${photo.image_url})`;
                infographicContainer.style.backgroundImage = newBg;
                document.querySelectorAll('#thumbnail-selector img').forEach(i => i.classList.remove('border-brand-purple'));
                thumb.classList.add('border-brand-purple');
            });
            thumbnailSelector.appendChild(thumb);
        });

        layoutRadios.forEach(radio => {
            radio.addEventListener('change', () => {
                infographicContainer.classList.remove('layout-banner-bottom', 'layout-classic-overlay', 'layout-orange-theme');
                infographicContainer.classList.add(radio.value);
            });
        });

        function updateToggles() {
            const hbcVisible = toggleHbcLogoCheckbox.checked;
            const genowaVisible = toggleLogoCheckbox.checked;
            document.querySelectorAll('.hbc-logo-wrapper').forEach(el => el.style.display = hbcVisible ? 'block' : 'none');
            document.querySelectorAll('.genowa-logo-wrapper').forEach(el => el.style.display = genowaVisible ? 'block' : 'none');
            document.querySelectorAll('.logo-divider').forEach(el => el.style.display = (hbcVisible && genowaVisible) ? 'block' : 'none');
            document.querySelectorAll('.hashtag-container').forEach(el => el.style.display = toggleHashtagCheckbox.checked ? 'block' : 'none');
        }
        [toggleLogoCheckbox, toggleHbcLogoCheckbox, toggleHashtagCheckbox].forEach(checkbox => {
            checkbox.addEventListener('change', updateToggles);
        });
        updateToggles();

        captionInput.addEventListener('input', () => {
            const captionText = captionInput.value;
            document.querySelectorAll('.infographic-caption').forEach(el => el.textContent = captionText);
            const currentLength = captionText.length;
            charCounter.textContent = `${currentLength} / ${maxChars}`;
            charCounter.classList.toggle('text-red-600', currentLength > maxChars);
            const isCaptionValid = captionText.trim() !== '' && currentLength <= maxChars;
            downloadBtn.disabled = !isCaptionValid;
        });

        downloadBtn.addEventListener('click', function() {
            downloadBtn.textContent = 'Generating...';
            downloadBtn.disabled = true;
            setTimeout(() => {
                html2canvas(infographicContainer, { useCORS: true, allowTaint: true, scale: 2 }).then(canvas => {
                    const link = document.createElement('a');
                    link.download = 'genowa_project_<?php echo e($project['id']); ?>.png';
                    link.href = canvas.toDataURL('image/png');
                    link.click();
                    downloadBtn.textContent = 'Download as PNG';
                    downloadBtn.disabled = false;
                }).catch(error => {
                    console.error('Error generating canvas:', error);
                    downloadBtn.textContent = 'Download as PNG';
                    downloadBtn.disabled = false;
                });
            }, 500);
        });
    <?php endif; ?>
});
</script>

<?php include 'includes/footer.php'; ?>
